<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\ViewModel\Checkout;

use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\DataObject;
use Magento\Framework\DataObjectFactory;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\UrlInterface;
use Magento\Quote\Model\Quote;
use Magento\Store\Model\StoreManagerInterface;
use Novuna\Pbf\Model\Config\AppConfig;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteIdManager;
use Novuna\Pbf\Model\PaymentMethod\PaymentMethodAvailability;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;
use Novuna\Pbf\ViewModel\AbstractPbfWidgetProvider;

class PbfWidget extends AbstractPbfWidgetProvider
{
    protected PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    protected Json $serializer;
    protected CheckoutSession $checkoutSession;
    private StoreManagerInterface $storeManager;
    /**
     * @var \Magento\Quote\Api\Data\CartInterface|\Magento\Quote\Model\Quote
     */
    protected $quote;
    protected UrlInterface $urlBuilder;
    private DataObjectFactory $dataObjectFactory;
    private EventManager $eventManager;
    private ?QuoteIdManager $quoteIdManager;
    private PaymentMethodAvailability $paymentMethodAvailability;

    public function __construct(
        EventManager $eventManager,
        DataObjectFactory $dataObjectFactory,
        CheckoutSession $checkoutSession,
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        Json $serializer,
        UrlInterface $urlBuilder,
        QuoteIdManager $quoteIdManager,
        PaymentMethodAvailability $paymentMethodAvailability,
        StoreManagerInterface $storeManager
    ) {
        $this->dataObjectFactory = $dataObjectFactory;
        $this->eventManager = $eventManager;
        $this->checkoutSession = $checkoutSession;
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->serializer = $serializer;
        $this->urlBuilder = $urlBuilder;
        $this->quoteIdManager = $quoteIdManager;
        $this->paymentMethodAvailability = $paymentMethodAvailability;
        $this->storeManager = $storeManager;
    }

    protected function getQuote(): Quote
    {
        if (!$this->quote) {
            $this->quote = $this->checkoutSession->getQuote();
        }
        return $this->quote;
    }

    public function getScriptSrc($eventName = ''): ?string
    {
        return $this->pbfWidgetConfigProvider->getScriptSrc($eventName);
    }

    public function canShowWidget(): bool
    {
        if (!$this->pbfWidgetConfigProvider->isPayByFinanceEnabled()) {
            return false;
        }

        $quote = $this->getQuote();

        //Can be shown anytime except on dep2nd paying for deposit:
        if ($this->paymentMethodAvailability->isAfterCMDep2nd($quote)) {
            return false;
        }

        //Allow widget if finance is not yet set in quote:
        if (!$quote->getPbfFinanceEnabled()) {
            return true;
        }
        //Can be shown anytime except on dep2nd paying for deposit:
        return !$this->paymentMethodAvailability->isAfterCMDep2nd($quote);
    }


    public function getJsonDataType(): string
    {
        return 'data-cart-json';
    }

    /**
     * Retrieve checkout URL
     *
     * @return string
     * @codeCoverageIgnore
     */
    public function getCheckoutUrl()
    {
        return $this->storeManager->getStore()->getUrl('checkout');
    }

    private function saveConfigSettings(array $widgetConfig = [])
    {

        $this->checkoutSession->setPbfConfigData($widgetConfig);
    }

    public function isFinanceEnabled(): bool
    {
        return (bool)$this->getQuote()->getPbfFinanceEnabled();
    }

    public function generateCartConfig(): array
    {
        $quote = $this->getQuote();

        if (!$quote->getId()) {
            return [];
        }

        $items = [];
        foreach ($quote->getAllVisibleItems() as $item) {
            $items[] = [
                'product_id' => $item->getProductId(),
                'sku' => $item->getSku(),
                'type' => $item->getProductType(),
                'quantity' => $item->getQty(),
                'variant_id' => $item->getId(),
                'price' => (float)$item->getPriceInclTax(),
                'priceExVat' => floatval($item->getPrice()),
                'tax_amount' => floatval($item->getTaxAmount()),
                'discount_amount' => floatval($item->getTotalDiscountAmount()),
                'discount_tax_compensation_amount' => floatval($item->getDiscountTaxCompensationAmount()),
            ];
        }

        $financeAmount = abs(floatval($quote->getPbfBaseFinanceAmount()));
        $shippingAddress = $quote->getShippingAddress();

        $allDiscounts = $this->getAllDiscounts($quote);

        $grandTotal = $shippingAddress->getGrandTotal() + $financeAmount;

        $shippingPrice = floatval($this->checkoutSession->getShippingPriceInclTax());
        $shippingExclTax = floatval($this->checkoutSession->getShippingPrice());
        if (!$shippingAddress->getShippingMethod()) {
            $grandTotal += $shippingPrice;
        }
        $widgetConfig = [
            'items' => $items,
            'shippingExVat' => floatval($shippingExclTax),
            'shipping' => floatval($shippingPrice),
            'shippingTitle' => $shippingAddress->getShippingMethod() ?? 'method',
            'total_price' => floatval($grandTotal),
            'checkout_url' => $this->getCheckoutUrl(),
            'discounts' => ['all_discounts' => $allDiscounts->getDiscountAmount()],
            'discountsExVat' => ['all_discounts' => $allDiscounts->getdiscountAmountBase()],
            'useDepSecond' => $quote->getPbfDeposit2nd(), // we need this for thank you page
            'cmVersion' => $this->pbfWidgetConfigProvider->getCmBackendVersion(),

            //We do not need these on Widget side:
            /*
            'pbfApplied' => (bool)$this->isFinanceEnabled(),
            'serverVersion' => $this->pbfWidgetConfigProvider->getPbfServerVersion(),
            'deposit_capture_type' => $this->pbfWidgetConfigProvider->depositCaptureType(),
            'checkoutToken' => $this->quoteIdManager->getMasked($quote),
            'isDraftOrder' => $this->isDraftOrderQuote($quote),

            'finance_active' => $quote->getPbfFinanceEnabled(),
*/
        ];

        //Do we need this?
        //$this->saveConfigSettings($widgetConfig); //save settings used to render widget on session for further use

        return $widgetConfig;
    }

    /**
     * @return string
     */
    public function getJsonConfig(): string
    {
        $quote = $this->getQuote();

        if (!$quote->getId()) {
            return $this->serializer->serialize([]);
        }
        $shippingAddress = $quote->getShippingAddress();
        $financeAmount = abs(floatval($quote->getPbfFinanceAmount()));
        $grandTotal = $shippingAddress->getGrandTotal() + $financeAmount;
        if ($shippingAddress->getShippingMethod()) {
            $shippingPrice = $shippingAddress->getShippingInclTax();
        } else {
            $shippingPrice = $this->checkoutSession->getShippingPriceInclTax();
            $grandTotal += $shippingPrice;
        }

        return $this->serializer->serialize(
            [
                'financeEnabled' => (bool)$this->isFinanceEnabled(),
                'shipping' => $shippingPrice,
                'total' => $grandTotal
            ]
        );
    }

    /**
     * @param $quote
     * @return DataObject
     */
    public function getAllDiscounts($quote): DataObject
    {
        $shippingAddress = $quote->getShippingAddress();

        $discountAmount = abs(floatval($shippingAddress->getDiscountAmount()));
        $discountAmountBase = abs(floatval($shippingAddress->getBaseDiscountAmount()));

        //commerce version support
        $discountAmount = $discountAmount + abs(floatval($quote->getCustomerBalanceAmountUsed()))
            + abs(floatval($quote->getGiftCardsAmountUsed()));
        $discountAmountBase = $discountAmountBase + abs((float)$quote->getBaseCustomerBalanceAmountUsed())
            + abs(floatval($quote->getBaseGiftCardsAmountUsed()));

        if ($discountAmount < 0) {
            $discountAmount = $discountAmountBase = 0;
        }
        $discount = $this->dataObjectFactory->create()->setData(
            ['discount_amount' => $discountAmount, 'discount_amount_base' => $discountAmountBase]
        );

        //retailers can use this observer to update total amount if they are using custom totals like store credit etc
        $this->eventManager->dispatch(
            'pbf_all_discounts',
            [
                'quote' => $quote,
                'discount' => $discount
            ]
        );

        return $discount;
    }

    private function isDraftOrderQuote(Quote $quote): bool
    {
        if (!$quote->getData(AppConfig::DRAFT_ORDER_FLAG)) {
            return false;
        }

        if ($quote->getData(AppConfig::DRAFT_ORDER_CONVERTED_FLAG)) {
            return false;
        }

        return true;
    }
}
