<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Ui\Component\Sales\Grid\Column;

use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\UiComponent\ContextInterface;
use Magento\Framework\View\Element\UiComponentFactory;
use Magento\Ui\Component\Listing\Columns\Column;
use Novuna\Pbf\Api\OrderRiskRepositoryInterface;
use Novuna\Pbf\Model\OrderRisk;
use Novuna\Pbf\Model\Sales\Service\FraudCheckService;

class PbfFraud extends Column
{
    private UrlInterface $urlBuilder;

    private FraudCheckService $fraudCheckService;
    private OrderRiskRepositoryInterface $orderRiskRepository;

    public function __construct(
        ContextInterface $context,
        UiComponentFactory $uiComponentFactory,
        UrlInterface $urlBuilder,
        FraudCheckService $fraudCheckService,
        OrderRiskRepositoryInterface $orderRiskRepository,
        array $components = [],
        array $data = []
    ) {
        $this->urlBuilder = $urlBuilder;
        $this->fraudCheckService = $fraudCheckService;
        $this->orderRiskRepository = $orderRiskRepository;
        parent::__construct($context, $uiComponentFactory, $components, $data);
    }

    public function prepareDataSource(array $dataSource)
    {
        if (isset($dataSource['data']['items'])) {
            foreach ($dataSource['data']['items'] as &$item) {
                if (isset($item['pbf_enabled']) && $item['pbf_enabled']) {
                    $orderId = (int)$item['entity_id'];
                    $fraudChecked = (bool)$item['pbf_fraud_checked'];
                    //Special case for `declined`:
                    if (isset($item['pbf_state_ro'])) {
                        $pbfState = $item['pbf_state_ro'];
                        if ($pbfState == FraudCheckService::PBF_DECLINED_STATE) {
                            $message = 'Finance Declined';
                            $item[$this->getData('name')] = "<span>$message</span>";
                            continue;
                        }
                    }

                    $risks = $this->orderRiskRepository->getOrderRisks($orderId);

                    $colorCode = $this->getPbfFraudColorCode($fraudChecked, $risks);
                    $message = $this->getPbfFraudMessage($fraudChecked, $risks);
                    $item[$this->getData('name')] = "<span style='color:" . $colorCode . " '>$message</span>";
                } else {
                    $item[$this->getData('name')] = "";
                }
            }
        }

        return $dataSource;
    }

    private function getPbfFraudColorCode($pbfFraudChecked, $risks): string
    {
        if ($pbfFraudChecked == false) {
            return '#303030';
        }

        if ($risks->count() == 0) {
            return '#075917FF';
        }

        return '#99221696';
    }

    private function getPbfFraudMessage($pbfFraudChecked, $risks): string
    {
        $pbfFraudStatusMessage
            = $this->fraudCheckService->getPbfFraudStatusMessage(
            $pbfFraudChecked, $risks, false
        );
        if (count($pbfFraudStatusMessage) == 1) {
            return $pbfFraudStatusMessage[0];
        }
        return "unknown";
    }
}
