<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Plugin\Sales\Model;

use Magento\Sales\Api\Data\OrderInterface;
use Novuna\Pbf\Api\OrderRiskRepositoryInterface;
use Novuna\Pbf\Model\OrderRisk;
use Novuna\Pbf\Model\Sales\Service\FraudCheckService;

class Order
{
    private FraudCheckService $fraudCheckService;
    private OrderRiskRepositoryInterface $orderRiskRepository;

    public function __construct(
        OrderRiskRepositoryInterface $orderRiskRepository,
        FraudCheckService $fraudCheckService
    ) {
        $this->fraudCheckService = $fraudCheckService;
        $this->orderRiskRepository = $orderRiskRepository;
    }

    /**
     * OOB shipment processing disabled if order has PBF fraud state
     * @param OrderInterface $order
     * @param $result
     * @return bool|mixed
     */
    public function afterCanShip(OrderInterface $order, $result)
    {
        if ($order->getPbfFinanceEnabled()) {
            if ($result && !$order->getSkipFraudCheck()) {
                $pbfFraudChecked = (bool)$order->getData(OrderRisk::FRAUD_CHECKED);
                $risks = $this->orderRiskRepository->getOrderRisks((int)$order->getId());
                $pbfUserAcptedRisks = (bool)$order->getData(OrderRisk::USER_ACCEPTED_FRAUD);
                $result = $this->fraudCheckService->canShip($pbfFraudChecked, $risks, $pbfUserAcptedRisks);
            }
        }
        return $result;
    }
}
