<?php

namespace Novuna\Pbf\Model\Queue;

use Magento\Framework\MessageQueue\EnvelopeFactory;
use Magento\Framework\MessageQueue\EnvelopeInterface;
use Magento\MysqlMq\Model\QueueManagement;
use Psr\Log\LoggerInterface;

/**
 * Do not understand why M2 thinks queue is "owned" by consumer: see vendor/magento/framework-message-queue/Consumer.php:240
 * If consumer finds message with topic that does not know, it calls $queue->reject() on that message?
 * What if there is another consumer that would process that message?
 * How M2 sees the decoupling purpose of queue (publisher side) and consumer side ?
 *
 * That's why we need to rewrite completely Queue in order to allow retry with backoff - that would normally be
 * consumer side change only, as it's related to message processing, not publishing/storing.
 *
 * There is probably missing piece similar to "Subscription" in this concept...
 */
class Queue extends \Magento\MysqlMq\Model\Driver\Queue
{
    private $topicSubscribed = [];

    protected $envelopeFactory;

    protected $queueName;

    private \Magento\MysqlMq\Model\ResourceModel\Queue $messageResource;
    
    /**
     * Queue constructor.
     *
     * @param QueueManagement $queueManagement
     * @param EnvelopeFactory $envelopeFactory
     * @param LoggerInterface $logger
     * @param string $queueName
     * @param int $interval
     * @param int $maxNumberOfTrials
     */
    public function __construct(
        QueueManagement $queueManagement,
        EnvelopeFactory $envelopeFactory,
        LoggerInterface $logger,
        \Magento\MysqlMq\Model\ResourceModel\Queue $messageResource,
        string $queueName,
        $interval = 5,
        $maxNumberOfTrials = 3
    ) {
        parent::__construct($queueManagement, $envelopeFactory, $logger, $queueName, $interval, $maxNumberOfTrials);
        $this->messageResource = $messageResource;
        $this->envelopeFactory = $envelopeFactory;
        $this->queueName = $queueName;
    }

    public function setMaxNumberOfTrials($n)
    {
        //PHP cannot lift private props visibility:
        $refl = new \ReflectionObject($this);
        $refl = $refl->getParentClass();
        if ($refl->getName() !== \Magento\MysqlMq\Model\Driver\Queue::class) {
            $refl = $refl->getParentClass();
            if ($refl->getName() !== \Magento\MysqlMq\Model\Driver\Queue::class) {
                throw new \Exception("cannot set MaxNumberOfTrials: not expected class hierarchy");
            }
        }

        $prop = $refl->getProperty('maxNumberOfTrials');
        $prop->setAccessible(true);
        $prop->setValue($this, $n);

        return $this;
    }

    public function subscribeToTopics($topics)
    {
        $this->topicSubscribed = $topics;
    }

    /**
     * This is where we changed behaviour only: we filter messages from database having topic in mind
     * and find first message that's backoff state allows processing right now.
     * This should be done in some sort of "Subscription" that would have all that niggles set and used.
     */
    public function dequeue()
    {
        $envelope = null;
        $messages = $this->readMessages(1);
        if (isset($messages[0])) {
            $properties = $messages[0];

            $body = $properties[QueueManagement::MESSAGE_BODY];
            unset($properties[QueueManagement::MESSAGE_BODY]);

            $envelope = $this->envelopeFactory->create(['body' => $body, 'properties' => $properties]);
        }

        return $envelope;
    }

    public function readMessages($maxMessages)
    {
        $selectedMessages = $this->getMessages(null); //Do not limit, we cannot represent backoff rules to SQL
        $selectedMessages = $this->filterWithEligibleBackOff($selectedMessages, $maxMessages);

            /* The logic below allows to prevent the same message being processed by several consumers in parallel */
        $selectedMessagesRelatedIds = [];
        foreach ($selectedMessages as &$message) {
            /* Set message status here to avoid extra reading from DB after it is updated */
            $message[\Magento\MysqlMq\Model\QueueManagement::MESSAGE_STATUS] = \Magento\MysqlMq\Model\QueueManagement::MESSAGE_STATUS_IN_PROGRESS;
            $selectedMessagesRelatedIds[] = $message[\Magento\MysqlMq\Model\QueueManagement::MESSAGE_QUEUE_RELATION_ID];
        }
        $takenMessagesRelationIds = $this->messageResource->takeMessagesInProgress($selectedMessagesRelatedIds);
        if (count($selectedMessages) == count($takenMessagesRelationIds)) {
            return $selectedMessages;
        } else {
            $selectedMessages = array_combine($selectedMessagesRelatedIds, array_values($selectedMessages));
            return array_intersect_key($selectedMessages, array_flip($takenMessagesRelationIds));
        }
    }

    private function filterWithEligibleBackOff($messages, $maxMessages)
    {
        $result = [];
        $cnt = 0;
        for ($i=0; $i < count($messages); $i++) {
            $properties = $messages[$i];
            $updatedAt =  new \DateTime($properties['updated_at']);
            $retries = $properties['retries'];
            if ($this->messageBackOffAllowProcess($updatedAt, $retries)) {
                $result[] = $properties;
                $cnt++;
                if ($cnt >= $maxMessages) {
                    break;
                }
            }

        }
        return $result;
    }

    private function messageBackOffAllowProcess($updatedAt, $retries):bool {
        if ($retries == 0) {
            return true;
        }

        $now = new \DateTime();

        $diffInSeconds = $now->getTimestamp() - $updatedAt->getTimestamp();
        switch ($retries) {
            case 1: return ($diffInSeconds > 10);
            case 2: return ($diffInSeconds > 60);
            case 3: return ($diffInSeconds > 5 * 60); //5 minutes
            case 4: return ($diffInSeconds > 10 * 60);
            case 5: return ($diffInSeconds > 30 * 60);
            case 6: return ($diffInSeconds > 2 * 60 * 60); //2hrs
            default: return ($diffInSeconds > 24 * 60 * 60); //each 24hrs
        }
    }

    public function getMessages($limit = null)
    {
        $connection = $this->messageResource->getConnection();
        $select = $connection->select()
            ->from(
                ['queue_message' => $this->messageResource->getTable('queue_message')],
                [QueueManagement::MESSAGE_TOPIC => 'topic_name', QueueManagement::MESSAGE_BODY => 'body']
            )->join(
                ['queue_message_status' => $this->messageResource->getTable('queue_message_status')],
                'queue_message.id = queue_message_status.message_id',
                [
                    QueueManagement::MESSAGE_QUEUE_RELATION_ID => 'id',
                    QueueManagement::MESSAGE_QUEUE_ID => 'queue_id',
                    QueueManagement::MESSAGE_ID => 'message_id',
                    QueueManagement::MESSAGE_STATUS => 'status',
                    QueueManagement::MESSAGE_UPDATED_AT => 'updated_at',
                    QueueManagement::MESSAGE_NUMBER_OF_TRIALS => 'number_of_trials'
                ]
            )->join(
                ['queue' => $this->messageResource->getTable('queue')],
                'queue.id = queue_message_status.queue_id',
                [QueueManagement::MESSAGE_QUEUE_NAME => 'name']
            )->where(
                'queue_message_status.status IN (?)',
                [QueueManagement::MESSAGE_STATUS_NEW, QueueManagement::MESSAGE_STATUS_RETRY_REQUIRED]
            )->where('queue.name = ?', $this->queueName)
            ->order(['queue_message_status.updated_at ASC', 'queue_message_status.id ASC']);

        if (count($this->topicSubscribed) > 0) {
            $select->where('queue_message.topic_name in (?)', $this->topicSubscribed);
        }
        
        if ($limit) {
            $select->limit($limit);
        }

        return $connection->fetchAll($select);
    }
}