<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\PaymentMethod;

use Novuna\Pbf\Model\Config\AppConfig;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteConverter;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteLocker;
use Magento\Quote\Model\Quote;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;

/**
 * Determines whether a payment method should be considered active or not
 * during checkout.
 *
 * This allows toggling of payment methods for deposit second i.e. when
 * deposit second is enabled and finance is chosen in the checkout then
 * hide all other payment methods.  If finance is not selected then show
 * all other payment methods and hide deposit second etc.
 *
 * can be static without any DI
 */
class PaymentMethodAvailability
{
    private QuoteConverter $quoteConverter;

    public function __construct(
        QuoteConverter $quoteConverter
    ) {
        $this->quoteConverter = $quoteConverter;
    }

    public function isAvailableForQuote(string $paymentMethodCode, Quote $quote): bool
    {
        //For non-finance checkouts or dep1st or Final dep2nd deposit checkout - show all except PBF payment method
        if (!$this->isFinanceQuote($quote) || $this->isDep1st($quote) || $this->isAfterCMDep2nd($quote)) {
            return $paymentMethodCode !== AppConfig::PAYMENT_METHOD_CODE;
        }

        //Dep2nd Checkout before CM3: show only PBF payment method
        return ($paymentMethodCode === AppConfig::PAYMENT_METHOD_CODE);
    }

    public function isDep1st(Quote $quote): bool
    {
        return $this->isFinanceQuote($quote) && !$quote->getPbfDeposit2nd();
    }

    public function isAfterCMDep2nd(Quote $quote): bool
    {
        if ($this->isOldDraftPaymentPending($quote)){
            return true;
        }
        return $this->isFinanceQuote($quote)
            && $quote->getPbfDeposit2nd()
            && $this->quoteConverter->isDraftOrderAvailableToConvert($quote); // This would be typical use of plain function call ;-( Needed to inject so many unused DI because of one call...
    }

    /**
     * @param Quote $quote
     * @return bool
     */
    private function isFinanceQuote(Quote $quote): bool
    {
        return (bool)$quote->getPbfFinanceEnabled();
    }

    /**
     * @param Quote $quote
     *
     * @return bool
     */
    private function isOldDraftPaymentPending(Quote $quote): bool
    {
        $oldIsDraftOrder = $quote->getData('draft_order');
        $oldIsConvertedDraftOrder = $quote->getData('draft_order_converted');

        return $oldIsDraftOrder && !$oldIsConvertedDraftOrder;
    }
}
