<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Controller;

use Laminas\Http\Client\Adapter\Curl;
use Laminas\Http\Header\HeaderInterface;
use Laminas\Http\Client;
use Laminas\Uri\Uri;
use Magento\Framework\App\Request\Http;
use Magento\Framework\Controller\Result\RawFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\View\Result\PageFactory;
use Magento\Store\Model\StoreManagerInterface;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;

class PbfWidget
{
    protected Http $request;
    protected StoreManagerInterface $storeManager;
    protected PageFactory $resultPageFactory;
    protected Uri $uri;
    protected RawFactory $resultRawFactory;
    protected PbfWidgetConfigProvider $pbfWidgetConfigProvider;

    /**
     * Constructor
     *
     * @param PbfWidgetConfigProvider $pbfWidgetConfigProvider
     * @param PageFactory $resultPageFactory
     * @param Http $request
     * @param RawFactory $resultRawFactory
     * @param StoreManagerInterface $storeManager
     * @param Uri $uri
     */
    public function __construct(
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        PageFactory $resultPageFactory,
        Http $request,
        RawFactory $resultRawFactory,
        StoreManagerInterface $storeManager,
        Uri $uri
    ) {
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->resultPageFactory = $resultPageFactory;
        $this->request = $request;
        $this->resultRawFactory = $resultRawFactory;
        $this->storeManager = $storeManager;
        $this->uri = $uri;
    }

    /**
     * Execute view action
     * @return ResultInterface
     * @throws NoSuchEntityException
     * @todo  add signature to requests
     */
    public function execute(): ResultInterface
    {

        $proxiedPath = PbfWidgetConfigProvider::PROXIED_PATH;
        $destinationPath = PbfWidgetConfigProvider::PROXY_DESTINATION_PATH;

        $uri = str_replace($proxiedPath, '', $this->request->getRequestUri());
        $parsedUri = $this->uri->parse($uri);
        $path = $parsedUri->getPath();
        //Shop param has to be included in request already, as Magento store config is so dynamic, that we cannot get store from URL
        $params = $parsedUri->getQueryAsArray();

        $url = $this->pbfWidgetConfigProvider->getAppServerUrl() . $destinationPath . $path . "?" . http_build_query($params);
        $requestBody = $this->getRequestBody();

        $http = new Client();
        $http->setAdapter(new Curl());
        $http->setUri($url);
        $http->setMethod('POST');
        $http->setRawBody($requestBody);
        $response = $http->send();

        $statusCode = $response->getStatusCode();
        $contentType = $response->getHeaders()->get('Content-Type');
        $content = $response->getBody();

        $result = $this->resultRawFactory->create();

        if ($contentType) {
            if ($contentType instanceof \ArrayIterator) {
                $contentType = $contentType[0];
            }

            if ($contentType instanceof HeaderInterface) {
                $contentType = $contentType->getFieldValue();
            }

            $result->setHeader('Content-Type', $contentType);
        }

        $result->setHttpResponseCode($statusCode);

        return $result->setContents($content);
    }

    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }

    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }

    protected function getRequestBody()
    {
        return $this->request->getContent();
    }
}
