<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\ViewModel\Product\View;

use Magento\Catalog\Model\Product;
use Magento\Framework\Registry;
use Magento\Framework\Serialize\Serializer\Json;
use Novuna\Pbf\Model\Entity\Attribute\Source\EnablePbf;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;
use Novuna\Pbf\ViewModel\AbstractPbfWidgetProvider;
use Magento\Checkout\Model\Session as CheckoutSession;


class PbfWidget extends AbstractPbfWidgetProvider
{
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    private Registry $coreRegistry;
    private Json $serializer;
    private CheckoutSession $checkoutSession;

    public function __construct(
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        Json $serializer,
        Registry $coreRegistry,
        CheckoutSession $checkoutSession
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->serializer = $serializer;
        $this->coreRegistry = $coreRegistry;
    }

    /**
     * Retrieve current Product object
     *
     * @return Product|null
     */
    public function getProduct(): ?Product
    {
        return $this->coreRegistry->registry('current_product');
    }

    public function canShowWidget(): bool
    {
        if (!$this->pbfWidgetConfigProvider->isPayByFinanceEnabled()) {
            return false;
        }

        $quote = $this->checkoutSession->getQuote();

        // prevent for legacy orders
        if ($quote && $quote->getFinanceEnabled()) {
            return false; 
        }

        $product = $this->getProduct();
        return $product->isSaleable() && $product->getPaybyfinanceEnable() != EnablePbf::VALUE_FORCE_DISABLE;
    }

    public function getScriptSrc($eventName = ''): ?string
    {
        return $this->pbfWidgetConfigProvider->getScriptSrc($eventName);
    }

    /**
     * @return string
     */
    public function getJsonDataType(): string
    {
        return 'data-product-json';
    }

    /**
     * @return string
     */
    public function getJsonConfig(): string
    {
        /* @var $product \Magento\Catalog\Model\Product */
        $product = $this->getProduct();
        $config = [
            'id' => $product->getId(),
            'price' => (float)$product->getFinalPrice(),
            'tags' => [[]],
            'variants' => [["id" => 0]]
        ];

        if ($product->getPaybyfinanceEnable() == EnablePbf::VALUE_FORCE_ENABLE) {
            $config['tags'] = [["PBF-Eligible"]];
        }

        if ($product->getPaybyfinanceEnable() == EnablePbf::VALUE_FORCE_DISABLE) {
            $config['tags'] = [["Not-eligible-for-PaybyFinance"]];
        }

        return $this->serializer->serialize($config);
    }
}
