<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Model;

use Magento\Framework\App\Config\ReinitableConfigInterface;
use Novuna\Pbf\Model\ResourceModel\StoreConfig;
use Magento\Framework\App\Cache\TypeListInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Encryption\Encryptor;

class StoreConfigProvider
{
    const XML_PATH_PAYBYFINANCE_ENABLED = 'pbf/general/enabled';
    const XML_PATH_PAYBYFINANCE_INSTALL_SCOPE = 'pbf/general/installation_scope';
    const XML_PATH_PBF_WIDGET_SCRIPT_SRC = 'pbf/widget/script_src';
    const XML_PATH_PBF_GENERAL_SETTINGS = 'pbf/general/settings';
    const XML_PATH_PBF_INSTALL_HMAC = 'pbf/install/hmac';
    const XML_PATH_PAYBYFINANCE_MODE = 'pbf/general/mode';
    const XML_PATH_PAYBYFINANCE_SALES_RULE_ID = 'pbf/salesrule/id';
    const XML_PATH_PAYBYFINANCE_FRAUD_MESSAGE = 'pbf/fraud_check/status_';
    const PBF_INTEGRATION_IDENTIFIER = 'PbfIntegration';
    const DELETE_INSTALL_HMAC_PARAM = 'instCleanup';
    /**
     * PBF MODES
     */
    const MODE_TEST = 'test';
    const MODE_LIVE = 'live';

    private ScopeConfigInterface $scopeConfig;
    private Encryptor $encryptor;
    private StoreConfig $configResource;
    private TypeListInterface $cacheTypeList;
    private ReinitableConfigInterface $appConfig;

    public function __construct(
        ScopeConfigInterface $scopeConfig,
        ReinitableConfigInterface $reInitableConfig,
        StoreConfig $configResource,
        TypeListInterface $cacheTypeList,
        Encryptor $encryptor
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->encryptor = $encryptor;
        $this->configResource = $configResource;
        $this->cacheTypeList = $cacheTypeList;
        $this->appConfig = $reInitableConfig;
    }

    /**
     * Check PBF enabled or not
     *
     * @return bool
     */
    public function isPayByFinanceEnabled($scopeType = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeCode = null): bool
    {
        return $this->scopeConfig->isSetFlag(StoreConfigProvider::XML_PATH_PAYBYFINANCE_ENABLED, $scopeType, $scopeCode);
    }

    /**
     * @return void
     */
    public function enablePayByFinance($isEnabled, $scope, $scopeId)
    {
        $this->saveConfig(self::XML_PATH_PAYBYFINANCE_ENABLED, $isEnabled, $scope, $scopeId);
    }

    /**
     * @return mixed
     */
    public function getPbfWidgetScriptSrc($eventName = '', $scopeType = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeCode = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_PBF_WIDGET_SCRIPT_SRC . '_' . $eventName, $scopeType, $scopeCode);
    }

    public function getPbfInstallHmac($scopeType = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeCode = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_PBF_INSTALL_HMAC, $scopeType, $scopeCode);
    }

    public function savePbfInstallHmac($hmac, $scope, $scopeId)
    {
        return $this->saveConfig(self::XML_PATH_PBF_INSTALL_HMAC, $hmac, $scope, $scopeId);
    }

    /**
     * @param $configId
     * @return mixed
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getPbfScriptTagById($configId)
    {
        return $this->configResource->getConfigById($configId);
    }

    /**
     * @return mixed
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getPbfScriptTags($scopeType = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeCode = null)
    {
        return $this->configResource->getConfig(
            self::XML_PATH_PBF_WIDGET_SCRIPT_SRC,
            $scopeType,
            $scopeCode,
            true
        );
    }

    /**
     * @throws \Exception
     */
    public function decrypt($value): string
    {
        return $this->encryptor->decrypt($value);
    }

    public function encrypt($value): string
    {
        return $this->encryptor->encrypt($value);
    }

    public function saveConfig($path, $value, $scopeType = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeCode = 0)
    {
        $configId = $this->configResource->saveConfig(
            $path,
            $value,
            $scopeType,
            $scopeCode
        );
        $this->cacheTypeList->cleanType('config');
        $this->appConfig->reinit();
        return $configId;
    }

    public function deleteConfigById($configId)
    {
        $this->configResource->deleteConfigById($configId);
        $this->cacheTypeList->cleanType('config');
        $this->appConfig->reinit();
    }

    /**
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function deleteConfigByPaths($paths = [], $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeId = 0)
    {
        $this->configResource->deleteConfigByPaths($paths, $scope, $scopeId);
        $this->cacheTypeList->cleanType('config');
        $this->appConfig->reinit();
    }

    public function getSettings($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeId = 0): ?array
    {
        $settings = $this->configResource->getConfig(
            self::XML_PATH_PBF_GENERAL_SETTINGS,
            $scope,
            $scopeId
        );

        if ($settings && is_string($settings)) {
            $settings = json_decode($settings, true);
        }

        return $settings ?: null;
    }

    public function setSettings(array $settings, $scopeType = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeCode = 0)
    {
        $this->saveConfig(
            self::XML_PATH_PBF_GENERAL_SETTINGS,
            json_encode($settings),
            $scopeType,
            $scopeCode
        );
    }

    public function setPbfDiscountRuleId($ruleId)
    {
        return $this->saveConfig(self::XML_PATH_PAYBYFINANCE_SALES_RULE_ID, $ruleId);
    }

    /**
     * @return mixed
     */
    public function getPbfDiscountRuleId()
    {
        return $this->scopeConfig->getValue(self::XML_PATH_PAYBYFINANCE_SALES_RULE_ID);
    }

    /**
     * @return mixed
     */
    public function getPbfMode()
    {
        return $this->scopeConfig->getValue(self::XML_PATH_PAYBYFINANCE_MODE);
    }
}
