<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\Api;

use Magento\Catalog\Model\ResourceModel\Product\Collection;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Framework\Webapi\Exception as WebapiException;
use Novuna\Pbf\Api\Data;
use Novuna\Pbf\Api\Data\ProductMetaInterface as ProductMetaApiInterface;
use Novuna\Pbf\Api\ProductMetaInterface;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Model\Data\ProductMetaData;
use Novuna\Pbf\Model\Data\ProductMeta as ProductMetaApiData;

class ProductMeta implements ProductMetaInterface
{
    const PBFNOTELIGIBLE = "Not-eligible-for-PaybyFinance";
    const PBFELIGIBLE = "PBF-Eligible";
    private StoreConfigProvider $storeConfigProvider;
    private ProductCollectionFactory $productCollectionFactory;

    public function __construct(
        StoreConfigProvider $storeConfigProvider,
        ProductCollectionFactory $productCollectionFactory
    ) {
        $this->storeConfigProvider = $storeConfigProvider;
        $this->productCollectionFactory = $productCollectionFactory;
    }

    /**
     * Query PBF to validate product meta data
     * @param string[] $productIds
     * @return ProductMetaApiInterface|null
     * @throws \Magento\Framework\Webapi\Exception
     * @throws \Magento\Framework\Exception\StateException
     */
    public function getProductMetaData(
        array $productIds = []
    ): ?ProductMetaApiInterface {
        if (!count($productIds)) {
            throw new WebapiException(__('Product Ids missing'));
        }
        $productsMetaData = [];
        foreach ($this->getProductCollection($productIds) as $product) {
            $tags = [];
            $isPbfEnabled = $product->getPaybyfinanceEnable();
            if ($isPbfEnabled == 2) {
                $tags[] = self::PBFNOTELIGIBLE;
            } elseif ($isPbfEnabled == 1) {
                $tags[] = self::PBFELIGIBLE;
            } else {
                $tags[] = "";
            }
            $productsMetaData[] = new ProductMetaData((int)$product->getId(), $tags);
        }
        return new ProductMetaApiData($productsMetaData);
    }

    /**
     * @param $productIds
     * @return Collection
     * @throws WebapiException
     */
    public function getProductCollection($productIds): Collection
    {
        $productCollection = $this->productCollectionFactory->create();
        $productCollection->addFieldToFilter('entity_id', ['in' => $productIds])
            ->addFieldToSelect(['paybyfinance_enable', 'price']);

        return $productCollection;
    }
}
