<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Controller\SetFinance;

use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\Http;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\View\Result\PageFactory;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\Quote;
use Magento\Store\Model\StoreManagerInterface;
use Novuna\Pbf\Logger\PbfLoggerInterface;
use Novuna\Pbf\Model\Config\AppConfig;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;

class Index implements HttpPostActionInterface, CsrfAwareActionInterface
{

    private Http $request;
    private JsonFactory $resultJsonFactory;
    private StoreManagerInterface $storeManager;
    private Json $serializer;
    private CheckoutSession $checkoutSession;
    /**
     * @var \Magento\Quote\Api\Data\CartInterface|Quote
     */
    private $quote;
    private CartRepositoryInterface $quoteRepository;
    private PbfLoggerInterface $logger;

    /**
     * Constructor
     *
     * @param Http $request
     * @param JsonFactory $resultJsonFactory
     * @param Json $serializer
     * @param CheckoutSession $checkoutSession
     * @param CartRepositoryInterface $quoteRepository
     * @param StoreManagerInterface $storeManager
     * @param PbfLoggerInterface $logger
     */
    public function __construct(
        Http $request,
        JsonFactory $resultJsonFactory,
        Json $serializer,
        CheckoutSession $checkoutSession,
        CartRepositoryInterface $quoteRepository,
        StoreManagerInterface $storeManager,
        PbfLoggerInterface $logger
    ) {
        $this->request = $request;
        $this->resultJsonFactory = $resultJsonFactory;
        $this->serializer = $serializer;
        $this->checkoutSession = $checkoutSession;
        $this->quoteRepository = $quoteRepository;
        $this->storeManager = $storeManager;
        $this->logger = $logger;
    }

    /**
     * Execute view action
     *
     * @return ResultInterface
     */
    public function execute(): ResultInterface
    {
        try {
            $contents = $this->request->getContent();
            $financeData = $this->serializer->unserialize($contents);
            $applyPbfData = $financeData['applyPbfData'] ?? [];

            $isFinanceEnabled = $financeData['finance'] ?? false;
            $depositSecond = $financeData['depositSecond'] ?? false;
            $financeAmount = ($isFinanceEnabled && isset($financeData['financeAmountIncVat'])) ?
                abs(floatval($financeData['financeAmountIncVat'])) : 0.00;
            $financeAmountExclVat = ($isFinanceEnabled && isset($financeData['financeAmount'])) ?
                abs(floatval($financeData['financeAmount'])) : 0.00;
            $this->checkoutSession->setFinanceData($financeData);

            $currentQuote = $this->getQuote();
            if (!$currentQuote->getData(AppConfig::DRAFT_ORDER_FLAG)) {

                $this->checkoutSession->setPbfFinanceEnabled($isFinanceEnabled);
                $this->checkoutSession->setPbfDeposit2nd($depositSecond);
                $currentQuote->setPbfFinanceEnabled($isFinanceEnabled);
                $currentQuote->setPbfDeposit2nd($depositSecond);

                if (
                    $currentQuote->getId()
                    && isset($financeData['contextPage'])
                    // && strpos($financeData['contextPage'], 'checkout') !== false
                ) {
                    $this->checkoutSession->setPbfFinanceAmount(-$financeAmount);

                    $currentQuote->setPbfFinanceAmount(-$financeAmount);
                    $currentQuote->setPbfFinanceExclTaxAmount(-$financeAmountExclVat);
                    $currentQuote->setPbfBaseFinanceAmount(-$financeAmount);
                    $currentQuote->setPbfAdditionalData($this->serializer->serialize($applyPbfData));
                    $currentQuote->collectTotals();
                    $this->quoteRepository->save($currentQuote);
                }
            }
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }
        $responseContent = ['success' => true];
        $resultJson = $this->resultJsonFactory->create();
        return $resultJson->setData($responseContent);
    }

    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }

    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }

    protected function getQuote()
    {
        if (!$this->quote) {
            $this->quote = $this->checkoutSession->getQuote();
        }
        return $this->quote;
    }
}
