<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Controller\Adminhtml\Configuration;

use Magento\Backend\App\Action\Context;
use Magento\Backend\Model\View\Result\Page;
use Magento\Framework\App\Action\HttpGetActionInterface as HttpGetActionInterface;
use Magento\Framework\Exception\IntegrationException;
use Magento\Framework\Phrase;
use Magento\Framework\Registry;
use Magento\Framework\View\Result\PageFactory;
use Magento\Integration\Api\IntegrationServiceInterface;
use Novuna\Pbf\Logger\PbfLoggerInterface;

class PermissionsDialog extends Base implements HttpGetActionInterface
{
    const ADMIN_RESOURCE = 'Novuna_Pbf::paybyfinance';
    /** Param Key for extracting integration id from Request */
    const PARAM_INTEGRATION_ID = 'id';

    const REGISTRY_KEY_CURRENT_INTEGRATION = 'current_integration';

    /** Saved API form data session key */
    const REGISTRY_KEY_CURRENT_RESOURCE = 'current_resource';
    private IntegrationServiceInterface $integrationService;
    private Registry $registry;
    private PbfLoggerInterface $logger;
    private PageFactory $resultPageFactory;

    public function __construct(
        PageFactory $resultPageFactory,
        Context $context,
        Registry $registry,
        IntegrationServiceInterface $integrationService,
        PbfLoggerInterface $logger
    ) {
        $this->registry = $registry;
        $this->integrationService = $integrationService;
        $this->logger = $logger;
        $this->resultPageFactory = $resultPageFactory;
        parent::__construct($context);
    }

    /**
     * Show permissions popup.
     *
     * @return void
     */
    public function execute()
    {
        $integrationId = (int)$this->getRequest()->getParam(self::PARAM_INTEGRATION_ID);
        if ($integrationId) {
            try {
                $integrationData = $this->integrationService->get($integrationId)->getData();
                $this->registry->register(self::REGISTRY_KEY_CURRENT_INTEGRATION, $integrationData);
            } catch (IntegrationException $e) {
                $this->messageManager->addErrorMessage($e->getMessage());
                $this->redirect();
                return;
            } catch (\Exception $e) {
                $this->logger->critical($e);
                $this->messageManager->addErrorMessage(__('Internal error. Check PBF log for details.'));
                $this->redirect();
                return;
            }
        } else {
            $this->messageManager->addErrorMessage(__('Integration ID is not specified or is invalid.'));
            $this->redirect();
            return;
        }

        $result = $this->resultPageFactory->create();
        if ($result instanceof Page) {
            $result->setActiveMenu(self::ADMIN_RESOURCE);
            $result->getConfig()->getTitle()->prepend(new Phrase('PBF - Access Authorization'));
        }

        return $result;
    }
}
