<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Block\Adminhtml\Order;

use Magento\Backend\Block\Widget\Context;
use Magento\Framework\Registry;
use Magento\Sales\Block\Adminhtml\Order\View as OrderView;
use Magento\Sales\Helper\Reorder;
use Magento\Sales\Model\ConfigInterface;
use Magento\Sales\Model\Order;
use Novuna\Pbf\Api\OrderRiskRepositoryInterface;
use Novuna\Pbf\Api\OrderRisksInterface;
use Novuna\Pbf\Model\OrderRisk;
use Novuna\Pbf\Model\Sales\Service\FraudCheckService;

class View extends OrderView
{
    private FraudCheckService $fraudCheckService;
    private OrderRiskRepositoryInterface $orderRiskRepository;

    public function __construct(
        Context $context,
        Registry $registry,
        ConfigInterface $salesConfig,
        Reorder $reorderHelper,
        FraudCheckService $fraudCheckService,
        OrderRiskRepositoryInterface $orderRiskRepository,
        array $data = []
    ) {
        $this->fraudCheckService = $fraudCheckService;
        $this->orderRiskRepository = $orderRiskRepository;
        parent::__construct(
            $context,
            $registry,
            $salesConfig,
            $reorderHelper,
            $data
        );
    }

    protected function _construct()
    {
        parent::_construct();

        $order = $this->getOrder();
        if ($order === null) {
            return;
        }
        $message = __('Are you sure you want to accept PBF risks flagged on the order?');
        if ($this->_isAllowedAction('Novuna_Pbf::fraud_authorization') && $this->canUpdatePbfFraudStatus($order)) {
            $this->addButton(
                'order_pbf_fraud_approve',
                [
                    'label' => __('Override PBF Warnings'),
                    'class' => 'fraud_approve',
                    'onclick' => "confirmSetLocation('{$message}', '{$this->getFraudAuthorizationUrl()}')",
                ]
            );
        }
    }

    private function getFraudAuthorizationUrl(): string
    {
        return $this->getUrl('pbf/order/acceptFraudStatus');
    }

    /**
     * Checks if order available for fraud status update
     *
     * @param Order $order
     * @return bool
     */
    public function canUpdatePbfFraudStatus(Order $order): bool
    {
        if (!$order->getPbfFinanceEnabled()) {
            return false;
        }

        if ($order->canUnhold() || $order->isPaymentReview()) {
            return false;
        }

        $state = $order->getState();
        if ($order->isCanceled() || $state === Order::STATE_CLOSED) {
            return false;
        }

        $pbfFraudChecked = (bool)$order->getData(OrderRisk::FRAUD_CHECKED);
        $risks = $this->orderRiskRepository->getOrderRisks((int)$order->getId());
        $pbfUserAcptedRisks = (bool)$order->getData(OrderRisk::USER_ACCEPTED_FRAUD);

        if ($this->fraudCheckService->canShip($pbfFraudChecked, $risks, $pbfUserAcptedRisks)) {
            return false;
        }

        return true;
    }
}
