<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Plugin\Sales\Model\Order;

use Magento\Sales\Model\OrderRepository;
use Novuna\Pbf\Helper\Data as DataHelper;
use Novuna\Pbf\Model\OrderRisk;
use Novuna\Pbf\Model\StoreConfigProvider;
use Magento\Sales\Api\Data\OrderInterface;
use Novuna\Pbf\Api\Data\WebhookDataInterface;
use Magento\Framework\HTTP\Client\CurlFactory;
use Novuna\Pbf\Api\WebhookRepositoryInterface;
use Magento\Sales\Api\Data\OrderAddressInterface;
use Novuna\Pbf\Model\Webhooks\WebhookSender;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;
use Novuna\Pbf\Model\Sales\Service\FraudCheckService;
use Novuna\Pbf\Controller\Adminhtml\Order\MassAcceptFraudStatus;

class Address
{
    private FraudCheckService $fraudCheckService;
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    private WebhookRepositoryInterface $webhookRepository;
    private CurlFactory $curlFactory;
    private OrderRepository $orderRepository;
    private WebhookSender $webhookSender;
    private \Magento\Sales\Model\ResourceModel\Order $orderResourceModel;

    public function __construct(
        FraudCheckService $fraudCheckService,
        WebhookRepositoryInterface $webhookRepository,
        CurlFactory $curlFactory,
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        OrderRepository $orderRepository,
        WebhookSender $webhookSender,
        \Magento\Sales\Model\ResourceModel\Order $orderResourceModel
    ) {
        $this->fraudCheckService = $fraudCheckService;
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->webhookRepository = $webhookRepository;
        $this->curlFactory = $curlFactory;
        $this->orderRepository = $orderRepository;
        $this->webhookSender = $webhookSender;
        $this->orderResourceModel = $orderResourceModel;
    }

    public function afterSave($resource, $result, $address)
    {
        $order = $this->orderRepository->get($address->getParentId());

        if ($address->getParentId() && $order && $order->getPbfFinanceEnabled()) {
            $oldAddressData = $this->getOrderAddressData($order, $address->getAddressType());
            if (empty(array_diff($address->getData(), $oldAddressData))) {
                return $result;
            }

            //Reset the fraud checked flag, as we changed address:
            $this->orderResourceModel->getConnection()->update($this->orderResourceModel->getMainTable(), [OrderRisk::FRAUD_CHECKED => 0], ['entity_id = ?' => $order->getId()] );

            $this->webhookSender->SendOrderMessage($order->getId(), WebhookSender::ORDER_UPDATE, null);
        }

        return $result;
    }

    private function getOrderAddressData($order, $addressType)
    {
        if ($addressType == "billing") {
            if ($order->getBillingAddress()) {
                return $order->getBillingAddress()->getData();
            }
            return [];
        }

        if ($addressType == "shipping") {
            if ($order->getShippingAddress()) {
                return $order->getShippingAddress()->getData();
            }
            return [];
        }

        return [];
    }
}
