<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Plugin\DraftOrder\Model;

use Novuna\Pbf\Model\DraftOrder\Quote\QuoteIdManager;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteLocker;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Quote\Model\Quote;
use Magento\Framework\App\RequestInterface as Request;
use Magento\Quote\Api\CartRepositoryInterface as CartRepository;
use Novuna\Pbf\Model\Config\AppConfig;

class QuoteLoader
{
    private Request $request;
    private CartRepository $cartRepository;
    private QuoteLocker $quoteLocker;
    private CheckoutSession $checkoutSession;
    private QuoteIdManager $quoteIdManager;

    public function __construct(
        CheckoutSession $checkoutSession,
        QuoteLocker $quoteLocker,
        QuoteIdManager $quoteIdManager,
        Request $request,
        CartRepository $cartRepository
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->quoteIdManager = $quoteIdManager;
        $this->quoteLocker = $quoteLocker;
        $this->request = $request;
        $this->cartRepository = $cartRepository;
    }

    public function beforeGetQuote(CheckoutSession $checkoutSession)
    {
        if ($this->request->getFullActionName() !== 'checkout_index_index') {
            return;
        }

        // If no param is set then we always want to double check it isnt a locked quote
        // and to reset if it is
        $quoteId = $this->request->getParam(AppConfig::INBOUND_URL_PARAMETER);

        if (!$quoteId) {
            $this->ensureCurrentQuoteIsNotLocked($checkoutSession);
        } else {
            $unmaskedQuoteId = null;
            if ($this->quoteIdManager->isIdMasked($quoteId)) {
                $unmaskedQuoteId = (int)$this->quoteIdManager->getUnMaskedById($quoteId);
            }

            $currentQuoteId = (int)$checkoutSession->getQuoteId();
            $draftOrderQuoteId = (int)$checkoutSession->getDraftOrderQuoteId();

            if ($unmaskedQuoteId && $unmaskedQuoteId != $draftOrderQuoteId) {
                $paramQuote = $this->cartRepository->get($unmaskedQuoteId);

                if ($this->isQuoteValid($paramQuote)) {
                    $paramQuote->setIsActive(true)->setReservedOrderId(null);
                    $this->cartRepository->save($paramQuote);

                    $checkoutSession->setQuoteId($paramQuote->getId());
                    $checkoutSession->setDraftOrderQuoteId($paramQuote->getId());

                    if ($currentQuoteId != $unmaskedQuoteId) {
                        if ($currentQuoteId != 0) {
                            $this->storeOldQuote($currentQuoteId);
                            $this->resetQuoteById($currentQuoteId);
                        }
                        $checkoutSession->replaceQuote($paramQuote);
                    }
                }
            }
        }
    }

    private function isQuoteValid(Quote $quote): bool
    {
        if (!$quote->getData(AppConfig::DRAFT_ORDER_FLAG)) {
            return false;
        }

        if ($quote->getData(AppConfig::DRAFT_ORDER_CONVERTED_FLAG)) {
            return false;
        }

        return true;
    }

    private function ensureCurrentQuoteIsNotLocked(CheckoutSession $checkoutSession)
    {
        $quoteId = $checkoutSession->getQuoteId();

        if ($quoteId) {
            $quote = $this->cartRepository->get($quoteId);
            if ($this->quoteLocker->isQuoteLocked($quote)) {
                $this->resetQuote($quote);
                $checkoutSession->setQuoteId(null);
            }
        } else {
            if ($oldQuoteId = $this->checkoutSession->getData('old_quote_id')) {
                $oldQuote = $this->cartRepository->get($oldQuoteId);
                if ($oldQuote) {
                    $oldQuote->setIsActive(true);
                    $this->cartRepository->save($oldQuote);
                    $this->checkoutSession->replaceQuote($oldQuote);
                }
            }
        }
    }

    /**
     * @param $quoteId
     * @return void
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function resetQuoteById($quoteId)
    {
        $quote = $this->cartRepository->get($quoteId);
        $this->resetQuote($quote);
    }

    /**
     * @param Quote $quote
     * @return void
     */
    private function resetQuote(Quote $quote)
    {
        $quote->setIsActive(0)->setReservedOrderId(null);
        $this->cartRepository->save($quote);
    }

    /**
     * @param $quoteId
     * @return void
     */
    private function storeOldQuote($quoteId)
    {
        $this->checkoutSession->setData('old_quote_id', $quoteId);
    }
}
