<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model;

use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Novuna\Pbf\Api\Data\DraftOrderDataInterface;
use Novuna\Pbf\Api\DraftOrderRepositoryInterface;
use Novuna\Pbf\Logger\PbfLoggerInterface;
use Novuna\Pbf\Model\ResourceModel\DraftOrder as ResourceDraftOrder;
use Novuna\Pbf\Model\ResourceModel\DraftOrder\Collection;
use Novuna\Pbf\Model\ResourceModel\DraftOrder\CollectionFactory as DraftOrderCollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class DraftOrderRepository implements DraftOrderRepositoryInterface
{

    protected ResourceDraftOrder $resource;

    protected DraftOrderFactory $draftOrderFactory;

    protected DraftOrderCollectionFactory $draftOrderCollectionFactory;

    private ?CollectionProcessorInterface $collectionProcessor;

    private SearchResultsInterface $searchResultsFactory;
    private PbfLoggerInterface $logger;

    /**
     * @param PbfLoggerInterface $logger
     * @param ResourceDraftOrder $resource
     * @param DraftOrderFactory $draftOrderFactory
     * @param DraftOrderCollectionFactory $draftOrderCollectionFactory
     * @param SearchResultsInterface $searchResultsFactory
     * @param CollectionProcessorInterface|null $collectionProcessor
     */
    public function __construct(
        PbfLoggerInterface $logger,
        ResourceDraftOrder $resource,
        DraftOrderFactory $draftOrderFactory,
        DraftOrderCollectionFactory $draftOrderCollectionFactory,
        SearchResultsInterface $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->logger = $logger;
        $this->resource = $resource;
        $this->draftOrderFactory = $draftOrderFactory;
        $this->draftOrderCollectionFactory = $draftOrderCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * Save DraftOrder data
     *
     * @param DraftOrderDataInterface $draftOrder
     *
     * @return DraftOrderDataInterface $draftOrder
     * @throws CouldNotSaveException
     */
    public function save(DraftOrderDataInterface $draftOrder)
    {
        try {
            $this->resource->save($draftOrder);
        } catch (\Exception $exception) {
            $this->logger->critical($exception->getMessage());
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $draftOrder;
    }

    /**
     * Load DraftOrder data by given ID
     *
     * @param $draftOrderId
     * @return DraftOrder
     * @throws NoSuchEntityException
     */
    public function getById($draftOrderId)
    {
        $draftOrder = $this->draftOrderFactory->create();
        $this->resource->load($draftOrder, $draftOrderId);
        if (!$draftOrder->getId()) {
            throw new NoSuchEntityException(
                __('Draft order "%1" ID doesn\'t exist.', $draftOrderId)
            );
        }

        return $draftOrder;
    }

    /**
     * @param string $quoteId
     * @return DraftOrderDataInterface
     */
    public function getByQuoteId($quoteId): DraftOrderDataInterface
    {
        $draftOrder = $this->draftOrderFactory->create();
        $this->resource->load($draftOrder, $quoteId, DraftOrderDataInterface::QUOTE_ID);
        return $draftOrder;
    }

    /**
     * @param string $orderId
     *
     * @return DraftOrderDataInterface
     */
    public function getByOrderId($orderId): DraftOrderDataInterface
    {
        $draftOrder = $this->draftOrderFactory->create();
        $this->resource->load($draftOrder, $orderId, DraftOrderDataInterface::ORDER_ID);
        return $draftOrder;
    }

    /**
     * @return Collection
     */
    public function getCollection()
    {
        return $this->draftOrderCollectionFactory->create();
    }

    /**
     * Load DraftOrder data collection by given search criteria
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     * @param SearchCriteriaInterface $searchCriteria
     * @return SearchResultsInterface
     */
    public function getList(SearchCriteriaInterface $searchCriteria): SearchResultsInterface
    {

        $collection = $this->getCollection();
        $this->collectionProcessor->process($searchCriteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * Delete DraftOrder
     *
     * @param DraftOrderDataInterface $draftOrder
     *
     * @return bool
     * @throws CouldNotDeleteException
     */
    public function delete(DraftOrderDataInterface $draftOrder): bool
    {
        try {
            $this->resource->delete($draftOrder);
        } catch (\Exception $exception) {
            $this->logger->critical($exception->getMessage());
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * Delete DraftOrder by given Identity
     *
     * @param string $draftOrderId
     *
     * @return bool
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    public function deleteById($draftOrderId): bool
    {
        return $this->delete($this->getById($draftOrderId));
    }
}
