<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Block\Adminhtml\Installation;

use Magento\Backend\Block\Template as BlockTemplate;
use Magento\Backend\Block\Template\Context as BlockContext;
use Magento\Backend\Model\Auth\Session;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Registry;
use Magento\Integration\Api\IntegrationServiceInterface;
use Magento\Integration\Api\OauthServiceInterface;
use Magento\Integration\Controller\Adminhtml\Integration as IntegrationController;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Store\Model\StoresConfig;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;
use Novuna\Pbf\Helper\Data as Helper;

class Widget extends BlockTemplate
{

    private Session $backendAuthSession;
    private StoreConfigProvider $storeConfigProvider;
    private IntegrationServiceInterface $integrationService;
    private Registry $coreRegistry;
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    private OauthServiceInterface $oauthService;

    public function __construct(
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        StoreConfigProvider $storeConfigProvider,
        BlockContext $context,
        Session $backendAuthSession,
        IntegrationServiceInterface $integrationService,
        Registry $registry,
        OauthServiceInterface $oauthService,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->backendAuthSession = $backendAuthSession;
        $this->storeConfigProvider = $storeConfigProvider;
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->integrationService = $integrationService;
        $this->coreRegistry = $registry;
        $this->oauthService = $oauthService;
    }

    public function isPBFInstalledOnScope(): bool
    {
        list($scopeType, $scopeId) = $this->pbfWidgetConfigProvider->getAdminReqScopeAndId();
        return (bool)$this->storeConfigProvider->isPayByFinanceEnabled($scopeType, $scopeId);
    }

    public function canDelete()
    {
        list($scopeType, $scopeId) = $this->pbfWidgetConfigProvider->getAdminReqScopeAndId();

        $pbfInstallScope = $this->_scopeConfig->getValue(StoreConfigProvider::XML_PATH_PAYBYFINANCE_INSTALL_SCOPE);
        if ($pbfInstallScope == null) {
            $pbfInstallScope = $scopeType;
        }

        return $scopeType == $pbfInstallScope && $this->isPBFInstalledOnScope();
    }

    public function isPBFIntegrationCreated(): bool
    {
        return $this->pbfWidgetConfigProvider->isPBFIntegrationCreated();
    }

    public function getCurrentIntegrationData(): array
    {
        $integrationData = $this->coreRegistry->registry(IntegrationController::REGISTRY_KEY_CURRENT_INTEGRATION);
        return $integrationData ?? [];
    }

    public function getCurrentIntegrationId()
    {
        return $this->getCurrentIntegrationData()['integration_id'] ?? null;
    }

    public function getConsumerCallbackUrl()
    {
        return $this->getRequest()->getParam('callback_url');
    }

    public function getConsumerReturnUrl()
    {
        return $this->getRequest()->getParam('return_url');
    }

    public function getCurrentIntegrationCallBackUrl(): string
    {
        return $this->getUrlWithScopeParams('pbf/configuration/acceptedSendToken');
    }

    public function getConfigurationsHomeUrl(): string
    {
        return $this->getUrlWithScopeParams('pbf/configuration/index');
    }

    public function getPermissionsSaveUrl(): string
    {
        return $this->getUrlWithScopeParams('pbf/configuration/createPermissions');
    }

    public function getPermissionsDeleteUrl(): string
    {
        return $this->getUrlWithScopeParams('pbf/configuration/uninstallPost');
    }

    public function getPbfUnInstallUrl(): string
    {
        return $this->getUrlWithScopeParams('pbf/configuration/uninstall');
    }

    private function getUrlWithScopeParams($route) {
        $params = [];
        $websiteId = $this->_request->getParam('website');
        if ($websiteId != "") {
            $params['website'] = $websiteId;
        }

        $storeId = $this->_request->getParam('store');
        if ($storeId != "") {
            $params['store'] = $storeId;
        }

        return $this->getUrl($route, $params);
    }

    public function getIframeUrl(): string
    {
        $params = [];
        $integrationName = StoreConfigProvider::PBF_INTEGRATION_IDENTIFIER;
        $integration = $this->integrationService->findByName($integrationName);
        $accessToken = $this->oauthService->getAccessToken($integration->getConsumerId());
        $secret = "";
        if ($accessToken) {
            $secret = $accessToken->getSecret();
        }
        $destinationPath = PbfWidgetConfigProvider::PROXY_DESTINATION_PATH;
        $now = new \DateTime();

        $params['timestamp'] = $now->getTimestamp();
        $params["shop"] = $this->pbfWidgetConfigProvider->getShopDomain();
        $params["hmac"] = $this->calculateHMAC($params, $secret);
        $params["sv"] = $this->pbfWidgetConfigProvider->getPbfServerVersion();
        return $this->pbfWidgetConfigProvider->getAppServerUrl() . $destinationPath . 'authenticate'
            . "?" . http_build_query($params);
    }

    /**
     * @todo
     * @return array
     */
    public function getIframeConfig(): array
    {
        return [];
    }

    private function calculateHMAC(array $params, $apiSecret)
    {
        return Helper::calculateHMAC($params, $apiSecret);
    }
}
