<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Plugin\Sales\Api;

use Magento\Sales\Api\Data\OrderExtensionInterface;
use Magento\Sales\Api\Data\OrderExtensionFactory;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\Data\OrderSearchResultInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Novuna\Pbf\Api\OrderRiskRepositoryInterface;
use Novuna\Pbf\Model\OrderRisk;
use Novuna\Pbf\Model\Sales\Service\FraudCheckService;

class OrderRepositoryPlugin
{

    private OrderExtensionFactory $orderExtensionFactory;
    private FraudCheckService $fraudCheckService;
    private OrderRiskRepositoryInterface $orderRiskRepository;

    public function __construct(
        FraudCheckService $fraudCheckService,
        OrderExtensionFactory $orderExtensionFactory,
        OrderRiskRepositoryInterface $orderRiskRepository
    ) {
        $this->fraudCheckService = $fraudCheckService;
        $this->orderExtensionFactory = $orderExtensionFactory;
        $this->orderRiskRepository = $orderRiskRepository;
    }

    protected function setExtensionAttributes(OrderInterface &$order)
    {
        $pbfAmount = $order->getData('finance_amount');
        $pbfFraudChecked = (bool)$order->getData(OrderRisk::FRAUD_CHECKED);
        $risks = $this->orderRiskRepository->getOrderRisks((int)$order->getId());
        $pbfUserAcptedRisks = (bool)$order->getData(OrderRisk::USER_ACCEPTED_FRAUD);
        $isPbfEnabled = $order->getData('finance_enabled');
        
        /** @var OrderExtensionInterface $extensionAttributes */
        $extensionAttributes = $order->getExtensionAttributes();
        $extensionAttributes = $extensionAttributes ? $extensionAttributes : $this->orderExtensionFactory->create();

        $extensionAttributes->setFinanceEnabled($isPbfEnabled);
        $extensionAttributes->setFinanceAmount($pbfAmount);
        $extensionAttributes->setPbfCanShip($this->fraudCheckService->canShip($pbfFraudChecked, $risks, $pbfUserAcptedRisks));

        $order->setExtensionAttributes($extensionAttributes);
    }

    /**
     *
     * @param OrderRepositoryInterface $subject
     * @param OrderInterface $order
     *
     * @return OrderInterface
     */
    public function afterGet(OrderRepositoryInterface $subject, OrderInterface $order): OrderInterface
    {
        $this->setExtensionAttributes($order);
        return $order;
    }

    public function afterGetList(
        OrderRepositoryInterface $subject,
        OrderSearchResultInterface $searchResult
    ): OrderSearchResultInterface {
        $orders = $searchResult->getItems();
        foreach ($orders as &$order) {
            $this->setExtensionAttributes($order);
        }

        return $searchResult;
    }
}
