<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Plugin\Quote\Model\Cart;

use Magento\Framework\Serialize\Serializer\Json;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\Data\TotalsInterface;
use Magento\Quote\Model\Cart\TotalSegment;

/*
 * This plugin is workarounding wrong implementation of M2 tax and
grand total being calculated partly in PHP and partly in JS.
in CartTotalRepository they are rewriting value in grandTotal:

        $amount = $quoteTotals->getGrandTotal() - $quoteTotals->getTaxAmount();
        $amount = $amount > 0 ? $amount : 0;
        $quoteTotals->setGrandTotal($amount);

which will break totals if there are some custom negative totals.

We are not fixing it, we are hacking and trying to punch some of its ramifications
*/
class CartTotalRepository
{
    private CartRepositoryInterface $quoteRepository;
    private Json $serializer;

    public function __construct(
        CartRepositoryInterface $quoteRepository,
        Json $serializer
    ) {
        $this->quoteRepository = $quoteRepository;
        $this->serializer = $serializer;
    }
 
    public function afterGet(
        \Magento\Quote\Model\Cart\CartTotalRepository $subject,
        TotalsInterface $quoteTotals,
        $cartId
    ) {
        $totalSegments = $quoteTotals->getTotalSegments() ?? [];
        if (isset($totalSegments['pbf_finance_amount'])
            && $totalSegments['pbf_finance_amount'] instanceof TotalSegment) {
            $financedAmount = abs(floatval($totalSegments['pbf_finance_amount']->getValue()));
            if ($financedAmount) {
                /** @var \Magento\Quote\Model\Quote $quote */
                $quote = $this->quoteRepository->getActive($cartId);
                //there are many async calls that can update grand total, so we cannot rely on data in it, we need to pass 0 deposit another way - 
                //tangling it all together ;-(
                $pbfData = $this->serializer->unserialize((string)$quote->getPbfAdditionalData());
                //For 0.deposit, we have to skip this plugin ;-(
                if (isset($pbfData['deposit']) and isset($pbfData['deposit']['percents'])) {
                    if ((int)$pbfData['deposit']['percents'] == 0) {
                        return $quoteTotals;
                    }
                }
                $amount = $quoteTotals->getGrandTotal() + $financedAmount -
                    abs(floatval($quote->getFinanceExclTaxAmount()));

                $amount = max($amount, 0);
                $quoteTotals->setGrandTotal($amount);
                $quoteTotals->setBaseGrandTotal($amount);
            }

        }
        return $quoteTotals;
    }
}
