<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Plugin\Checkout\Model;

use Magento\Checkout\Api\Data\PaymentDetailsInterface;
use Magento\Checkout\Api\Data\ShippingInformationInterface;
use Magento\Checkout\Api\ShippingInformationManagementInterface;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\Exception\InputException;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\Quote;

class ShippingInformationManagement
{
    const RESTRICTED_COUNTRY_ID = 'GB'; //What country is required for financed carts
    private CartRepositoryInterface $quoteRepository;
    private CheckoutSession $checkoutSession;

    public function __construct(
        CartRepositoryInterface $quoteRepository,
        CheckoutSession $checkoutSession
    ) {
        $this->quoteRepository = $quoteRepository;
        $this->checkoutSession = $checkoutSession;
    }

    public function beforeSaveAddressInformation(
        ShippingInformationManagementInterface $subject,
        $cartId,
        ShippingInformationInterface $addressInformation
    ) {
        /** @var Quote $quote */
        $quote = $this->quoteRepository->getActive($cartId);
        if ($quote && $quote->getFinanceEnabled()) {
            $this->validateAddress($addressInformation);
        }
        return [$cartId, $addressInformation];
    }

    /**
     * Update shipping total saved in session after we have correct shipping details
     * @param ShippingInformationManagementInterface $subject
     * @param PaymentDetailsInterface $paymentDetails
     * @return PaymentDetailsInterface $paymentDetails
     */
    public function afterSaveAddressInformation(
        ShippingInformationManagementInterface $subject,
        PaymentDetailsInterface $paymentDetails
    ): PaymentDetailsInterface {
        $totals = $paymentDetails->getTotals();
        $this->checkoutSession->setShippingPrice($totals->getShippingAmount());
        $this->checkoutSession->setShippingPriceInclTax(
            $totals->getShippingAmount() + $totals->getShippingTaxAmount()
        );
        return $paymentDetails;
    }

    /**
     * @param ShippingInformationInterface $addressInformation
     * @throws InputException
     */
    private function validateAddress(ShippingInformationInterface $addressInformation)
    {
        $address = $addressInformation->getShippingAddress();
        if ($address && $address->getCountryId() !== self::RESTRICTED_COUNTRY_ID) {
            throw new InputException(
                __(
                    'When financed, only UK address can be used for billing.'
                )
            );
        }
    }
}
