<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\Klarna\Orderline;

use Klarna\Core\Api\BuilderInterface;
use Klarna\Core\Model\Checkout\Orderline\AbstractLine;

// @codingStandardsIgnoreStart
if (interface_exists(BuilderInterface::class) && class_exists(AbstractLine::class)) {
    /**
     * Generate order line details for customer balance
     */
    class FinanceAmount extends AbstractLine
    {
        /**
         * Checkout item type
         */
        const ITEM_TYPE_CODE = 'pbf_finance_amount';

        /**
         * {@inheritdoc}
         */
        public function collect(BuilderInterface $checkout)
        {
            /** @var \Magento\Quote\Model\Quote $quote */
            $quote = $checkout->getObject();
            $totals = $quote->getTotals();

            if (!is_array($totals)) {
                return $this;
            }

            if (!isset($totals[self::ITEM_TYPE_CODE]) && $quote->getFinanceAmount()) {
                $totals[self::ITEM_TYPE_CODE] = $this->dataObjectFactory->create([
                    'data' => [
                        'value' => abs(floatval($quote->getFinanceAmount())),
                        'title' => 'Financed Amount',
                        'code' => self::ITEM_TYPE_CODE,
                    ]
                ]);
            }

            if (isset($totals[self::ITEM_TYPE_CODE])) {
                $total = $totals[self::ITEM_TYPE_CODE];
                $value = $this->helper->toApiFloat($total->getValue());

                $checkout->addData([
                    self::ITEM_TYPE_CODE . '_unit_price' => $value,
                    self::ITEM_TYPE_CODE . '_tax_rate' => 0,
                    self::ITEM_TYPE_CODE . '_total_amount' => $value,
                    self::ITEM_TYPE_CODE . '_tax_amount' => 0,
                    self::ITEM_TYPE_CODE . '_title' => $total->getTitle(),
                    self::ITEM_TYPE_CODE . '_reference' => $total->getCode()

                ]);
            }

            return $this;
        }

        /**
         * {@inheritdoc}
         */
        public function fetch(BuilderInterface $checkout)
        {
            if ($checkout->getData(self::ITEM_TYPE_CODE . '_total_amount')) {
                $checkout->addOrderLine([
                    'type' => self::ITEM_TYPE_CODE,
                    'reference' => $checkout->getData(self::ITEM_TYPE_CODE . '_reference'),
                    'name' => $checkout->getData(self::ITEM_TYPE_CODE . '_title'),
                    'quantity' => 1,
                    'unit_price' => $checkout->getData(self::ITEM_TYPE_CODE . '_unit_price'),
                    'tax_rate' => $checkout->getData(self::ITEM_TYPE_CODE . '_tax_rate'),
                    'total_amount' => $checkout->getData(self::ITEM_TYPE_CODE . '_total_amount'),
                    'total_tax_amount' => $checkout->getData(self::ITEM_TYPE_CODE . '_tax_amount'),
                ]);
            }

            return $this;
        }
    }
} else {
    class FinanceAmount
    {

    }
}
// @codingStandardsIgnoreEnd