<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\DraftOrder\Quote;

use Magento\Customer\Model\Group;
use Magento\Framework\DB\Transaction;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Quote\Api\CartManagementInterface;
use Magento\Quote\Api\CartRepositoryInterface as QuoteRepository;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\TotalsCollector;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;
use Magento\Sales\Model\Order;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Model\Quote\Address\ToOrderAddress;
use Magento\Sales\Model\Service\InvoiceService;
use Novuna\Pbf\Api\Data\DraftOrderDataInterface;
use Novuna\Pbf\Api\SalesOrderMetaRepositoryInterface;
use Novuna\Pbf\Logger\PbfLoggerInterface;
use Novuna\Pbf\Model\Api\OrderManagementService;
use Novuna\Pbf\Model\Config\AppConfig;
use Novuna\Pbf\Model\DraftOrderRepository;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Model\Webhooks\WebhookSender;
use Magento\Sales\Model\Order\Status\HistoryFactory;

class QuoteConverter
{
    private QuoteRepository $quoteRepository;
    private OrderRepository $orderRepository;
    private EventManager $eventManager;
    private DraftOrderRepository $draftOrderRepository;
    private PbfLoggerInterface $logger;
    private CartManagementInterface $quoteManagement;
    private InvoiceService $invoiceService;
    private TotalsCollector $totalsCollector;
    private Transaction $transaction;
    private StoreConfigProvider $storeConfigProvider;
    private SalesOrderMetaRepositoryInterface $metaRepository;
    private WebhookSender $webhookSender;
    private Order\Status\HistoryFactory $orderHistoryFactory;
    private OrderManagementService $managementService;
    private HistoryFactory $historyFactory;
    private ToOrderAddress $toOrderAddress;

    public function __construct(
        StoreConfigProvider $storeConfigProvider,
        DraftOrderRepository $draftOrderRepository,
        CartManagementInterface $quoteManagement,
        QuoteRepository $quoteRepository,
        OrderRepository $orderRepository,
        InvoiceService $invoiceService,
        Transaction $transaction,
        TotalsCollector $totalsCollector,
        EventManager $eventManager,
        SalesOrderMetaRepositoryInterface $metaRepository,
        PbfLoggerInterface $logger,
        WebhookSender $webhookSender,
        \Magento\Sales\Model\Order\Status\HistoryFactory $orderHistoryFactory,
        OrderManagementService $managementService,
        HistoryFactory $historyFactory,
        ToOrderAddress $toOrderAddress
    ) {
        $this->storeConfigProvider = $storeConfigProvider;
        $this->draftOrderRepository = $draftOrderRepository;
        $this->quoteRepository = $quoteRepository;
        $this->orderRepository = $orderRepository;
        $this->quoteManagement = $quoteManagement;
        $this->invoiceService = $invoiceService;
        $this->transaction = $transaction;
        $this->totalsCollector = $totalsCollector;
        $this->eventManager = $eventManager;
        $this->metaRepository = $metaRepository;
        $this->logger = $logger;
        $this->webhookSender = $webhookSender;
        $this->orderHistoryFactory = $orderHistoryFactory;
        $this->managementService = $managementService;
        $this->historyFactory = $historyFactory;
        $this->toOrderAddress = $toOrderAddress;
    }

    public function isDraftOrderAvailableToConvert(Quote $quote): bool
    {
        $isDraftOrder = $quote->getData(AppConfig::DRAFT_ORDER_FLAG);
        $isConvertedDraftOrder = $quote->getData(AppConfig::DRAFT_ORDER_CONVERTED_FLAG);
        return $isDraftOrder && !$isConvertedDraftOrder;
    }

    /**
     * @throws \Exception
     */
    public function convertDraftOrderToRealOrder(Quote $quote): Order
    {
        if ($this->isDraftOrderAvailableToConvert($quote)) {
            if (!$quote->getCustomerId()) {
                $quote->setCustomerId(0)
                    ->setCustomerIsGuest(true)
                    ->setCustomerGroupId(Group::NOT_LOGGED_IN_ID);
            }
            $quote->setPaymentMethod(AppConfig::PAYMENT_METHOD_CODE);
            $this->quoteRepository->save($quote);
            $quote->getPayment()->importData(['method' => AppConfig::PAYMENT_METHOD_CODE]);

            //@todo verify
            $total = $this->totalsCollector->collect($quote);
            $quote->addData($total->getData());

            $order = $this->quoteManagement->submit($quote);
            $this->connectDraftOrder($quote, $order);
            $order->setState('processing');
            $order->setStatus('processing');
            //CM3 took the deposit - there is nothing left to pay
            $this->createInvoice($order);

            return $order;
        }
        throw new \Exception("Cannot convert draft order to order, DraftOrder is not read fpr conversion.");
    }

    public function connectDraftOrder(Quote $quote, Order $order): Quote
    {

        if ($this->isDraftOrderAvailableToConvert($quote) && $order->getId()) {
            try {
                $draftOrder = $this->draftOrderRepository->getByQuoteId($quote->getId());
                if ($draftOrder->getId()) {
                    $quote->setData(AppConfig::DRAFT_ORDER_CONVERTED_FLAG, true);
                    $quote->setData(AppConfig::DRAFT_ORDER_CONVERTED_ID_FIELD, $order->getId());

                    $this->quoteRepository->save($quote);

                    $this->copyCommentsFromDraft($draftOrder, $order);

                    $billingAddress = $order->getBillingAddress();
                    foreach ($draftOrder->getBillingAddress()->getData() as $k => $v) {
                        $billingAddress->setData($k, $v);
                    }
                    $billingAddress->setCustomerAddressId(null);

                    $shippingAddress = $order->getShippingAddress();
                    foreach ($draftOrder->getShippingAddress()->getData() as $k => $v) {
                        $shippingAddress->setData($k, $v);
                    }
                    $shippingAddress->setCustomerAddressId(null);

                    $this->orderRepository->save($order);

                    $this->webhookSender->SendOrderMessage($order->getId(), WebhookSender::DRAFT_CONVERTED, ["quote_id" => (int)$quote->getId()]);

                    $this->eventManager->dispatch(
                        'draft_order_converted_to_order',
                        [
                            'draft_order' => $draftOrder,
                            'quote' => $quote,
                            'order' => $order
                        ]
                    );

                    $this->logger->info(__("Draft Order quote %s converted to order", $quote->getId()));
                } else {
                    $this->logger->error(__("Draft Order quote %s entry missing", $quote->getId()));
                }
            } catch (CouldNotSaveException $e) {
                $this->logger->critical($e);
            }
        }
        return $quote;
    }

    protected function copyCommentsFromDraft(DraftOrderDataInterface $draftOrder, Order $order)
    {
        //We should call service API here, but it's not possible in M2: $this->draftOrderService->getDraftComments()
        $comments = $this->getDraftComments($draftOrder->getQuoteId());
        $this->managementService->addComment((int)$order->getId(), $comments);
    }

    /**
     * This should be in API, but M2 does not allow two way reference. Or we should create business layer, which has not place in M2 neither
     * @param $quoteId
     *
     * @return array
     */
    public function getDraftComments($quoteId)
    {
        $quote = $this->quoteRepository->get($quoteId);
        $json_comments = $quote->getDraftComments();
        $rawComments = $json_comments ? json_decode($json_comments) : [];
        $comments = [];
        foreach ($rawComments as $rawComment) {
            $comment = $this->historyFactory->create();
            $comment->setComment($rawComment->comment);
            $comment->setParentId($rawComment->parent_id);
            $comment->setCreatedAt($rawComment->created_at);

            $comments[] = $comment;
        }
        return $comments;
    }

    /**
     * @param $order
     *
     * @throws CouldNotSaveException
     */
    private function createInvoice($order): void
    {
        try {
            $invoice = $this->invoiceService->prepareInvoice($order);
            $invoice->register();
            $transactionSave = $this->transaction->addObject($invoice)->addObject(
                $invoice->getOrder()
            );

            $transactionSave->save();
        } catch (\Exception $e) {
            $this->logger->critical($e);
            throw new CouldNotSaveException(__($e->getMessage()));
        }
    }
}
