<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\DraftOrder;

use Magento\Framework\Event\ManagerInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Novuna\Pbf\Api\Data\DraftOrderDataInterface;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteIdManager;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteLocker;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\Data\CartInterface;
use Novuna\Pbf\Model\DraftOrderFactory;
use Novuna\Pbf\Model\DraftOrderRepository;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;

class DraftOrderMaker
{
    private QuoteLocker $quoteLocker;
    private CartRepositoryInterface $cartRepository;
    private QuoteIdManager $quoteIdManager;
    private DraftOrderFactory $draftOrderFactory;
    private DraftOrderRepository $draftOrderRepository;
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    private ManagerInterface $eventManager;
    private OrderRepositoryInterface $orderRepository;

    public function __construct(
        QuoteLocker $quoteLocker,
        CartRepositoryInterface $cartRepository,
        QuoteIdManager $quoteIdManager,
        DraftOrderFactory $draftOrderFactory,
        OrderRepositoryInterface $orderRepository,
        DraftOrderRepository $draftOrderRepository,
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        ManagerInterface $eventManager
    ) {
        $this->quoteLocker = $quoteLocker;
        $this->cartRepository = $cartRepository;
        $this->quoteIdManager = $quoteIdManager;
        $this->draftOrderFactory = $draftOrderFactory;
        $this->draftOrderRepository = $draftOrderRepository;
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->eventManager = $eventManager;
        $this->orderRepository = $orderRepository;
    }

    public function makeDraftOrder(string $cid, $pbf_taking_deposit): DraftOrderDataInterface
    {
        if ($this->isIdMasked($cid)) {
            $cid = $this->quoteIdManager->getUnMaskedById($cid);
        }

        $quote = $this->loadQuote((int)$cid);
        $this->lockQuote($quote);
        return $this->generateDraftOrder($quote, $pbf_taking_deposit);
    }

    public function getDraftOrder(string $cid): DraftOrderDataInterface
    {
        if ($this->isIdMasked($cid)) {
            $cid = $this->quoteIdManager->getUnMaskedById($cid);
        }

        $quote = $this->loadQuote((int)$cid);
        $draftOrder = $this->draftOrderRepository->getByQuoteId($quote->getId());
        $draftOrder->setQuote($quote);
        return $draftOrder;
    }

    public function getDraftOrderByOrderId(int $orderId): DraftOrderDataInterface
    {
        $order = $this->orderRepository->get($orderId);
        $quote = $this->loadQuote((int)$order->getQuoteId());
        $draftOrder = $this->draftOrderRepository->getByQuoteId($quote->getId());
        $draftOrder->setQuote($quote);
        return $draftOrder;
    }

    public function isIdMasked(string $cid): bool
    {
        return $this->quoteIdManager->isIdMasked($cid);
    }

    private function loadQuote(int $cid): CartInterface
    {
        return $this->cartRepository->get($cid);
    }

    private function lockQuote(CartInterface $quote): CartInterface
    {
        $this->quoteLocker->lockQuote($quote);
        $this->cartRepository->save($quote);

        return $quote;
    }

    private function generateDraftOrder(CartInterface $quote, $pbf_taking_deposit): DraftOrderDataInterface
    {
        $draftOrder = $this->draftOrderRepository->getByQuoteId($quote->getId());

        $draftOrderData = [
            'quote_id' => $quote->getId(),
            'finance_enabled' =>$quote->getFinanceEnabled(),
            'pbf_additional_data' =>$quote->getPbfAdditionalData(),
            'finance_amount' => $quote->getFinanceAmount(),
            'subtotal' => $quote->getShippingAddress()->getSubtotalInclTax(),
            'grand_total' => $quote->getShippingAddress()->getGrandTotal(),
            'pbf_taking_deposit' => $pbf_taking_deposit
        ];
        $draftOrder->addData($draftOrderData);

        $draftOrder = $this->draftOrderRepository->save($draftOrder);
        $draftOrder->setQuote($quote);
        $this->eventManager->dispatch(
            'draft_order_save_after',
            ['order' => $draftOrder]
        );
        return $draftOrder;
    }
}
