<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Controller\Adminhtml\Configuration;

use Laminas\Uri\Uri;
use Magento\Backend\App\Action\Context;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\Escaper;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\HTTP\ZendClient;
use Magento\Framework\Oauth\Exception;
use Magento\Integration\Api\IntegrationServiceInterface;
use Magento\Integration\Api\OauthServiceInterface;
use Magento\Integration\Controller\Adminhtml\Integration as IntegrationController;
use Magento\Integration\Helper\Oauth\Data as IntegrationOauthHelper;
use Magento\Integration\Model\Integration as IntegrationModel;
use Novuna\Pbf\Logger\PbfLoggerInterface;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;
use Zend_Http_Client;
use Magento\Store\Model\ScopeInterface;

class AcceptedSendToken extends Base implements HttpPostActionInterface
{
    const ADMIN_RESOURCE = 'Novuna_Pbf::paybyfinance';
    private PbfLoggerInterface $logger;
    private Escaper $escaper;
    private OauthServiceInterface $oauthService;
    private IntegrationServiceInterface $integrationService;
    private ZendClient $httpClient;
    private IntegrationOauthHelper $dataHelper;
    private StoreConfigProvider $storeConfigProvider;
    private Uri $uri;
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;

    public function __construct(
        StoreConfigProvider $storeConfigProvider,
        OauthServiceInterface $oauthService,
        IntegrationServiceInterface $integrationService,
        ZendClient $httpClient,
        Uri $uri,
        IntegrationOauthHelper $dataHelper,
        Escaper $escaper,
        PbfLoggerInterface $logger,
        Context $context,
        PbfWidgetConfigProvider $pbfWidgetConfigProvider
    ) {
        $this->storeConfigProvider = $storeConfigProvider;
        $this->oauthService = $oauthService;
        $this->integrationService = $integrationService;
        $this->escaper = $escaper;
        $this->logger = $logger;
        $this->dataHelper = $dataHelper;
        $this->httpClient = $httpClient;
        $this->uri = $uri;
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        parent::__construct($context);
    }

    public function execute()
    {
        try {
            if ($this->isValidPostRequest()) {

                $integrationId = $this->getRequest()->getParam('integration_id');

                $integration = $this->integrationService->get($integrationId);

                $consumerId = $integration->getConsumerId();
                if ($this->oauthService->createAccessToken($consumerId, true)) {
                    $integration->setStatus(IntegrationModel::STATUS_ACTIVE)->save();
                    $accessToken = $this->oauthService->getAccessToken($consumerId);
                    $consumer = $this->oauthService->loadConsumer($consumerId);
                    $callbackURL = $this->getRequest()->getParam("callback_url");
                    $consumerData = $consumer->getData();
                    $this->sendAccessTokenToConsumer($callbackURL, $consumerData['key'], $consumerData['secret'],  $accessToken->getToken(), $accessToken->getSecret());
                    
                    list($scope, $scopeId) = $this->pbfWidgetConfigProvider->getAdminReqScopeAndId();
                    $params =  [IntegrationController::PARAM_INTEGRATION_ID => $integration->getId()];
                    if ($scope != '') {
                        $params[$scope] = $scopeId;
                    }
                    //Continue to installation fo PBF to keep it smooth:
                    return $this->_redirect(
                        'paybyfinance/configuration/installation',
                        $params
                    );
                } else {
                    $this->messageManager->addErrorMessage(__('Unable to create access Token'));
                }
            }
        } catch (AuthenticationException $e) {
            $this->logger->error($e . '\n' . $e->getTraceAsString());
            $this->messageManager->addErrorMessage($e->getMessage());
        } catch (LocalizedException $e) {
            $this->logger->error($e . '\n' . $e->getTraceAsString());
            $this->messageManager->addErrorMessage($this->escaper->escapeHtml($e->getMessage()));
        } catch (\Exception $e) {
            $this->logger->critical($e . '\n' . $e->getTraceAsString());
            $this->messageManager->addErrorMessage($this->escaper->escapeHtml($e->getMessage()));
        }
        $this->redirect();
    }

    /**
     * @throws Exception
     */
    private function sendAccessTokenToConsumer($endpointUrl, $consumer_key, $consumer_secret, $access_token, $access_token_secret)
    {
        try {
            $this->httpClient->setUri($endpointUrl);
            $this->httpClient->setParameterPost('consumer_key', $consumer_key);
            $this->httpClient->setParameterPost('consumer_secret', $consumer_secret);
            $this->httpClient->setParameterPost('access_token', $access_token);
            $this->httpClient->setParameterPost('access_token_secret', $access_token_secret);

            $maxRedirects = $this->dataHelper->getConsumerPostMaxRedirects();
            $timeout = $this->dataHelper->getConsumerPostTimeout();
            $this->httpClient->setConfig(['maxredirects' => $maxRedirects, 'timeout' => $timeout]);
            $this->httpClient->request(Zend_Http_Client::POST);
        } catch (\Exception $exception) {
            throw new \Magento\Framework\Oauth\Exception(
                __('The attempt to post data to consumer failed due to an unexpected error. Please try again later.'),
                $exception
            );
        }
    }
}
