<?php

namespace Mtc\WorldPay;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;
use Mtc\Modules\Members\Classes\Auth;
use Mtc\Modules\Members\Classes\Member;

/**
 * WorldPay Payment Gateway
 *
 * @package  Mtc\WorldPay
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class WorldPay implements PaymentGateway
{
    /**
     * Constants for WorldPay version to be used
     */
    const VERSION_FORM = 'form';
    const VERSION_INLINE = 'inline';

    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('worldpay.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        if (config('worldpay.version') === self::VERSION_FORM) {
            $template = (new WorldPayForm($invoice))->template();
            return new PaymentForm($template, 'template', [
                'name' => __('worldpay::worldpay.payment_option_name')
            ]);
        }

        // Inline version
        return new PaymentForm('worldpay-payment', 'vue-component', [
            'key' => config('worldpay.public_key'),
            'invoice_id' => $invoice->getId(),
            'verify_url' => route('worldpay.verify_url'),
            'payment_url' => route('worldpay.payment_url'),
            'name' => __('worldpay::worldpay.payment_option_name')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        if (config('worldpay.version') === self::VERSION_FORM) {
            return (new WorldPayForm($invoice))->charge($request);
        }

        // Inline form processing
        $charge_data = $this->chargeInvoice($request, $invoice);
        return [
            'provider' => 'worldpay',
            'amount' => $charge_data->amount / 100,
            'reference' => $charge_data->id,
            'details' => $charge_data,
            'confirmed_at' => now(),
        ];

    }

    /**
     * Charge Invoice outstanding amount
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     */
    public function chargeInvoice(Request $request, InvoiceRepositoryContract $invoice)
    {
        throw new Exception('Not implemented');
    }

}
