<?php

namespace Mtc\WorldPay;

use Illuminate\Http\Request;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\WorldPay\Http\Requests\WorldPayFormNotification;

/**
 * WorldPay Form integration
 *
 * @package  Mtc\WorldPay
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class WorldPayForm
{
    /**
     * Invoice object for payment
     *
     * @var InvoiceRepositoryContract
     */
    protected $invoice;

    /**
     * WorldPayForm constructor.
     *
     * @param InvoiceRepositoryContract $invoice
     */
    public function __construct(InvoiceRepositoryContract $invoice)
    {
        $this->invoice = $invoice;
    }

    /**
     * When asking for string render the template into form
     *
     * @return string
     */
    public function template()
    {
        return template('worldpay/payment_form.twig', [
            'invoice' => $this->invoice,
            'worldpay_url' => $this->getEndpoint(),
            'fields' => $this->getFields(),
        ]);
    }

    /**
     * Get the payment gateway endpoint based on live/test mode
     *
     * @return string
     */
    protected function getEndpoint()
    {
        $environment = config('worldpay.live') ? 'secure' : 'secure-test';
        return "https://{$environment}.worldpay.com/wcc/purchase?" . http_build_query([
                'successURL' => route('worldpay.return_url', [ $this->invoice->getId() ]),
            ]);

    }

    /**
     * Get the fields used in form
     *
     * @return array
     */
    protected function getFields()
    {
        $invoice_model =  $this->invoice->getModel();
        $address = collect([
            $invoice_model->recipient->address1,
            $invoice_model->recipient->address2,
            $invoice_model->recipient->city,
        ])->filter()->implode("\n");

        return [
            'signature' => $this->getSignature(),
            'instId' => config('worldpay.install_id'),
            'accId' => config('worldpay.account_id'),
            'authMode' => 'A',
            'cartId' => $this->invoice->getId(),
            'amount' => $this->invoice->getOutstandingAmount(),
            'currency' => $this->invoice->getCurrency(),
            'desc' => __('worldpay::worldpay.payment_description', [
                'site_name' => config('app.name'),
                'invoice_id' => $this->invoice->getId(),
            ]),
            'testMode' => config('worldpay.live') ? 0 : 101,
            'M_preorder' => $this->invoice->getPayable()->getAttribute('id'),
            'email' => $this->invoice->getEmail(),
            'tel' => $invoice_model->contact_number,
            'name' => $invoice_model->recipient->customer_name,
            'address' => $address,
            'country' => $invoice_model->recipient->country,
            'postcode' => $invoice_model->recipient->postcode,
            'successUrl' => route('worldpay.return_url', [ $this->invoice->getId() ]),
        ];
    }

    /**
     * Generate the signature string
     *
     * @return string
     */
    protected function getSignature()
    {
        return  md5(
            config('worldpay.secret_key') . ':'
            . $this->invoice->getCurrency() . ':'
            . $this->invoice->getOutstandingAmount() . ':'
            . (config('worldpay.live') ? 0 : 101) . ':'
            . config('worldpay.install_id')
        );
    }

    /**
     * Create a payment charge
     * Since this is done on WorldPay servers and we already validated the request through request class
     * we simply need to create a response array that will be used for saving the payment
     *
     * @param WorldPayFormNotification $request
     * @return array
     */
    public function charge(Request $request)
    {
        return [
            'provider' => 'worldpay',
            'amount' => $request->input('amount'),
            'reference' => $request->input('transId'),
            'details' => $request->input(),
            'confirmed_at' => now(),
        ];
    }
}
