<?php

namespace Mtc\VehicleValuation;

use App\Http\Requests\ValuationCustomerRequest;
use Carbon\Carbon;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Http;
use Mtc\VehicleValuation\Models\VehicleValuation;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;
use Mtc\VehicleValuation\Config\MotorCheckConfig;
use Mtc\MotorCheck\Services\MotorCheckService;
use Illuminate\Support\Facades\App;

class MotorCheck implements Contracts\ValuationDriver
{
    public function __construct(protected MotorCheckConfig $config)
    {
        Config::set('motor_check.specs_username', $config->clientId());
        Config::set('motor_check.specs_password', $config->clientSecret());

        Config::set('motor_check.enabled', $config->enabled());
        Config::set('motor_check.allow_local_cache', false);
        Config::set('motor_check.live', true);
        Config::set('motor_check.specs_live', true);

        Config::set('motor_check.valuation_driver', $config->valuationDriver());
    }

    public function getValuation(string $registration, int $mileage, ?string $variant = null): ValuationResponse
    {
        if ($this->config->valuationDriver() === 'identity-lookup') {
            $vehicleSpecification = $this->getIdentityLookupSpec($registration);
            $registrationDate = !empty($vehicleSpecification->registration_date)
                ? Carbon::createFromFormat('Y-m-d', $vehicleSpecification->registration_date)
                : Carbon::now();

            // Identity specs lookup does not have valuation, just vehicle lookup
            $valuation = new \Mtc\AutomotiveSpecs\VehicleValuation();
        } else {
            $motorCheck = App::make(MotorCheckService::class);
            $vehicleSpecification = $motorCheck->getCarSpecs($registration, $mileage);
            $registrationDate = !empty($vehicleSpecification->registration_date)
                ? Carbon::createFromFormat('Y-m-d', $vehicleSpecification->registration_date)
                : Carbon::now();

            $valuation = $motorCheck->getValuation($registration, $mileage);

            if (empty($valuation->retail)) {
                throw new ValuationRequestFailed('Failed to get valuation details from MotorCheck');
            }
        }

        return new ValuationResponse(
            provider: 'motor-check',
            registration: $registration,
            mileage: $mileage,
            vehicleType: '',
            retailPrice: $valuation->retail,
            averagePrice: null,
            cleanPrice: null,
            belowPrice: null,
            make: $vehicleSpecification->make ?? '',
            model: $vehicleSpecification->model ?? '',
            derivative: $vehicleSpecification->derivative ?? '',
            fuelType: $vehicleSpecification->fuel_type ?? '',
            engineSize: $vehicleSpecification->engine_size ?? '',
            bodyType: $vehicleSpecification->body_style ?? '',
            dateOfValuation: now(),
            transmission: $vehicleSpecification->transmission ?? '',
            registrationDate: $registrationDate,
            colour: $vehicleSpecification->colour ?? '',
        );
    }

    public function sendCustomerData(ValuationCustomerRequest $request, VehicleValuation $valuation): bool
    {
        return true;
    }

    private function getIdentityLookupSpec(string $registration): object
    {
        $response = Http::get($this->identityEndpoint($registration));
        $vehicleData = simplexml_load_string($response->body());

        return (object)([
            'make' => (string)$vehicleData->vehicle->make,
            'model' => (string)$vehicleData->vehicle->model,
            'derivative' => (string)$vehicleData->vehicle->version,
            'fuel_type' => (string)$vehicleData->vehicle->fuel,
            'engine_size' => (string)$vehicleData->vehicle->engine_cc ?: null,
            'body_style' => (string)$vehicleData->vehicle->body,
            'transmission' => (string)$vehicleData->vehicle->transmission === 'A' ? 'Automatic' : 'Manual',
            'registration_date' => (string)$vehicleData->vehicle->reg_date,
            'colour' => (string)$vehicleData->vehicle->colour,
        ]);
    }

    private function identityEndpoint(string $registration): string
    {
        $domain = match ($this->config->country()) {
            'IE' => 'ie',
            default => 'com'
        };
        return "https://api.motorcheck.$domain/vehicle/reg/$registration/identity/vin?"
            . http_build_query([
                "_username" => $this->config->clientId(),
                "_api_key" => $this->config->clientSecret(),
            ]);
    }
}
