<?php

namespace Mtc\VehicleValuation;

use App\Http\Requests\ValuationCustomerRequest;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\App;
use Mtc\VehicleValuation\Contracts\HasFutureValuations;
use Mtc\VehicleValuation\Contracts\HasValuationConfirmation;
use Mtc\VehicleValuation\Contracts\MultiStepValuation;
use Mtc\VehicleValuation\Contracts\ValuationDriver;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Models\VehicleValuation;

class ValuationRepository
{
    protected ValuationDriver $driver;

    public function hasMultiStep(): bool
    {
        return $this->getApiDriver() instanceof MultiStepValuation;
    }

    public function getVehicleVariants(string $registration_number, int $mileage)
    {
        return $this->getApiDriver()->getVariants($registration_number, $mileage);
    }

    /**
     * Get valuation
     *
     * @param string $registration
     * @param int $mileage
     * @param string|null $variant
     * @param string|null $cosmetic_condition
     * @return Builder|Model|VehicleValuation|object|null
     */
    public function get(string $registration, int $mileage, ?string $variant = null, ?string $cosmetic_condition = null)
    {
        if (!empty($variant)) {
            return $this->store($this->getApiDriver()->getValuation($registration, $mileage, $variant, $cosmetic_condition));
        }
        return $this->getLocalValuation($registration, $mileage)
            ?? $this->store($this->getApiDriver()->getValuation($registration, $mileage, $variant, $cosmetic_condition));
    }

    public function sendCustomerData(ValuationCustomerRequest $request, VehicleValuation $valuation): bool
    {
        return $this->getApiDriver()->sendCustomerData($request, $valuation);
    }

    public function hasEnabledDriver(): bool
    {
        return collect(config('valuation.drivers'))
                ->filter(fn($driverData) => App::make($driverData['config'])->enabled())
                ->map(fn($driverData) => new $driverData['class'](App::make($driverData['config'])))
                ->count() > 0;
    }

    /**
     * Get a locally stored valuation before performing api lookup
     *
     * @param string $registration
     * @param int $mileage
     * @return Builder|Model|object|null
     */
    protected function getLocalValuation(string $registration, int $mileage): ?VehicleValuation
    {
        return $this->getQuery()
            ->where('registration', $registration)
            ->where('mileage', $mileage)
            ->where('valuation_made_at', '>=', Carbon::now()->subDays(config('valuation.max_request_age')))
            ->whereNull('errors')
            ->first();
    }

    /**
     * Get the active valuation driver
     *
     * @return ValuationDriver|HasFutureValuations
     */
    protected function getApiDriver(): ValuationDriver
    {
        return isset($this->driver) ? $this->driver : $this->initializeDriver();
    }

    /**
     * Store a valuation result from API
     *
     * @param ValuationResponse $valuation
     * @return VehicleValuation|null
     */
    protected function store(ValuationResponse $valuation): ?VehicleValuation
    {
        $data = $valuation->toArray();

        // don't store or return invalid valuations
        if (empty($data['make'])) {
            return null;
        }

        if (empty($data['valuation_made_at'])) {
            $data['valuation_made_at'] = Carbon::now();
        }
        $valuation = $this->getQuery()->create($data);

        $driver = $this->getApiDriver();
        if ($driver instanceof HasValuationConfirmation) {
            $driver->confirmValuation($valuation);
        }

        return $valuation;
    }

    /**
     * Initialize valuation driver
     *
     * @return ValuationDriver
     */
    protected function initializeDriver(): ValuationDriver
    {
        return collect(config('valuation.drivers'))
            ->filter(fn($driverData) => App::make($driverData['config'])->enabled())
            ->map(fn($driverData) => new $driverData['class'](App::make($driverData['config'])))
            ->firstOrFail();
    }

    protected function getQuery(): Builder
    {
        return VehicleValuation::query();
    }
}
