<?php

namespace Mtc\VehicleValuation;

use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleValuation\Config\CapConfig;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;

class CapData implements Contracts\ValuationDriver
{
    public function __construct(protected CapConfig $config)
    {
        //
    }

    public function getValuation(string $registration, int $mileage): ValuationResponse
    {
        try {
            $response = $this->config->client->post($this->endpoint('/VRMValuation'), [
                'form_params' => [
                    'SubscriberID' => $this->config->subscriberId(),
                    'Password' => $this->config->password(),
                    'VRM' => $registration,
                    'Mileage' => $mileage,
                    'StandardEquipmentRequired' => 'false'
                ]
            ]);

            $xml = simplexml_load_string((string)$response->getBody());

            if ((string)$xml->Success === 'true') {
                return new ValuationResponse(
                    provider: 'cap-data',
                    registration: $registration,
                    mileage: (string)$xml->Valuation->ValuationMileage,
                    vehicleType: 'car',
                    make: (string)$xml->VRMLookup->CAPMan,
                    model: (string)$xml->VRMLookup->CAPMod,
                    derivative: (string)$xml->VRMLookup->CAPDer,
                    retailPrice: (string)$xml->Valuation->Retail,
                    cleanPrice: (string)$xml->Valuation->Clean,
                    averagePrice: (string)$xml->Valuation->Average,
                    belowPrice: (string)$xml->Valuation->Below,
                );
            }
            Log::info('CapData Valuation Request was not successful', [
                $xml
            ]);
        } catch (GuzzleException $exception) {
            Log::warning('CapData Valuation Request failed', [
                'message' => $exception->getMessage(),
                'registration' => $registration,
                'mileage' => $mileage,
            ]);
        }
        throw new ValuationRequestFailed('CapData Valuation request failed');
    }

    /**
     * Api endpoint for valuation
     *
     * @param string $path
     * @return string
     */
    protected function endpoint(string $path): string
    {
        return 'https://soap.cap.co.uk/Vrm/CapVrm.asmx' . $path;
    }
}
