<?php

namespace Mtc\VehicleValuation;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\App;
use Mtc\VehicleValuation\Contracts\ValuationDriver;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Models\VehicleValuation;

class ValuationRepository
{
    protected ValuationDriver $driver;

    /**
     * Get valuation
     *
     * @param string $registration
     * @param int $mileage
     * @return VehicleValuation|Model
     */
    public function get(string $registration, int $mileage)
    {
        return $this->getLocalValuation($registration, $mileage)
            ?? $this->store($this->getApiDriver()->getValuation($registration, $mileage));
    }

    /**
     * Get a locally stored valuation before performing api lookup
     *
     * @param string $registration
     * @param int $mileage
     * @return \Illuminate\Database\Eloquent\Builder|Model|object|null
     */
    protected function getLocalValuation(string $registration, int $mileage): ?VehicleValuation
    {
        return VehicleValuation::query()
            ->where('registration', $registration)
            ->where('mileage', $mileage)
            ->where('valuation_made_at', '>=', Carbon::now()->subDays(config('valuation.max_request_age')))
            ->whereNull('errors')
            ->first();
    }

    /**
     * Get the active valuation driver
     *
     * @return ValuationDriver
     */
    protected function getApiDriver(): ValuationDriver
    {
        return isset($this->driver) ? $this->driver : $this->initializeDriver();
    }

    /**
     * Store a valuation result from API
     *
     * @param ValuationResponse $valuation
     * @return \Illuminate\Database\Eloquent\Builder|Model
     */
    protected function store(ValuationResponse $valuation): VehicleValuation
    {
        return VehicleValuation::query()->create($valuation->toArray());
    }

    /**
     * Initialize valuation driver
     *
     * @return ValuationDriver
     */
    protected function initializeDriver(): ValuationDriver
    {
        return collect(config('valuation.drivers'))
            ->filter(fn ($driverData) => App::make($driverData['config'])->enabled())
            ->map(fn ($driverData) => new $driverData['class'](App::make($driverData['config'])))
            ->firstOrFail();
    }
}
