<?php

namespace Mtc\VehicleValuation;

use App\Http\Requests\ValuationCustomerRequest;
use Carbon\Carbon;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleValuation\Models\VehicleValuation;
use Mtc\VehicleValuation\Config\BeepBeepConfig;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;

class BeepBeep implements Contracts\ValuationDriver
{
    public function __construct(protected BeepBeepConfig $config)
    {
    }

    public function getValuation(
        string $registration,
        int $mileage,
        ?string $variant = null,
        ?string $cosmetic_condition = null,
    ): ValuationResponse {
        $vehicleData = $this->getVehicleData($registration);

        return new ValuationResponse(
            provider: 'beepbeep',
            registration: $registration,
            mileage: $mileage,
            vehicleType: '',
            retailPrice: null,
            averagePrice: null,
            cleanPrice: null,
            belowPrice: null,
            make: $vehicleData['vehicleSummary']['make']
                ?? $vehicleData['ukHistory']['technical']['make'] ?? '',
            model: $vehicleData['vehicleSummary']['model'] ?? $vehicleData['ukHistory']['technical']['model'] ?? '',
            derivative: $vehicleData['vehicleSummary']['modelDescription'] ?? '',
            fuelType: $vehicleData['vehicleSummary']['fuelType']
                ?? $vehicleData['ukHistory']['technical']['fuel'] ?? '',
            engineSize: $vehicleData['vehicleSummary']['engineCapacity']
                ?? $vehicleData['ukHistory']['technical']['engineCapacity'] ?? '',
            bodyType: $vehicleData['vehicleSummary']['bodyType']
                ?? $vehicleData['ukHistory']['technical']['bodyType'] ?? '',
            dateOfValuation: now(),
            transmission: $vehicleData['vehicleSummary']['transmission']
                ?? $vehicleData['ukHistory']['technical']['transmission'] ?? '',
            registrationDate: Carbon::createFromFormat(
                'Y-m-d\TH:i:s',
                $vehicleData['vehicleSummary']['dateFirstReg']
                ?? $vehicleData['ukHistory']['technical']['dateFirstReg']
            ),
            colour: $vehicleData['vehicleSummary']['colour'] ?? $vehicleData['ukHistory']['technical']['colour'] ?? '',
        );
    }

    /**
     * @param string $registration
     * @return array|null
     * @throws ValuationRequestFailed
     */
    public function getVehicleData(string $registration): ?array
    {
        return Http::withHeaders($this->headers())
            ->post($this->endpoint(), [
                'registration' => $registration
            ])
            ->onError(function (Response $response) use ($registration) {
                Log::warning('BeepBeep lookup failed to get derivative ID', [
                    'message' => $response->body(),
                    'registration' => $registration,
                ]);
                throw new ValuationRequestFailed('BeepBeep was not able to find vehicle: '
                    . $response->json('message'));
            })->json();
    }

    public function sendCustomerData(ValuationCustomerRequest $request, VehicleValuation $valuation): bool
    {
        return true;
    }

    protected function headers(): array
    {
        return [
            'Authorization' => "Bearer " . $this->config->apiKey(),
        ];
    }

    /**
     * Api endpoint for valuation
     *
     * @return string
     */
    protected function endpoint(): string
    {
        return 'https://member.carhistorycheck.ie/api/lookup';
    }
}
