<?php

namespace Mtc\VehicleValuation;

use App\Http\Requests\ValuationCustomerRequest;
use Carbon\Carbon;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleValuation\Contracts\HasValuationConfirmation;
use Mtc\VehicleValuation\Models\VehicleValuation;
use Mtc\VehicleValuation\Config\BCAConfig;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;
use GuzzleHttp\Client;

class BCA implements Contracts\MultiStepValuation, HasValuationConfirmation
{
    public function __construct(protected BCAConfig $config)
    {
        $this->config = $config;
    }

    public function getValuation(
        string $registration,
        int $mileage,
        ?string $variant = null,
        ?string $cosmetic_condition = null,
    ): ValuationResponse {
        $vehicleData = $this->getVehicleData($registration);
        $vehicleSpecification = collect($vehicleData['vehicleSpecifications'])
            ->when(!empty($variant), fn ($collection) => $collection->where('identifier', $variant))
            ->first();
        if (!$vehicleSpecification) {
            throw new ValuationRequestFailed('Failed to get confirm vehicle variant.');
        }

        $valuation = Http::asJson()->withHeaders($this->headers())
            ->post($this->config->getEndpoint($registration) . 'valuation', [
                'registrationNumber' => $registration,
                'identifier' => $vehicleSpecification['identifier'],
                'odometerReading' => $mileage,
                'colour' => $vehicleSpecification['colour'],
                'registrationDate' => $vehicleSpecification['registrationDate'],
                'vehicleType' => $vehicleSpecification['vehicleType'],
                "Category" => "Lead",
                "correlationId" => $vehicleData['correlationId'],
                "answers" => [
                    "ownership" => "yes",
                    "motStatus" => "yes",
                    "insuranceLoss" => "no",
                    "taxiPrivateHire" => "no",
                    "emergencyServices" => "no",
                    "ukFirstRegistered" => "yes",
                    "beenModified" => "no",
                    "mechanicalCondition" => "yes",
                    "cosmeticCondition" => $this->getCosmeticCondition($cosmetic_condition),
                    "totalKeepers" => "1",
                    "serviceHistory" => "full",
                ],
                "vehicleSpecifications" => array_filter($vehicleSpecification, function ($item) {
                    return !in_array($item, [
                        'registrationNumber',
                        'identifier',
                        'vehicleType',
                        'colour',
                        'registrationDate',
                    ]);
                })
            ])
            ->onError(function (Response $response) use ($registration, $mileage) {
                Log::warning('BCA Valuation Request failed', [
                    'message' => $response->body(),
                    'status' => $response->status(),
                    'registration' => $registration,
                    'mileage' => $mileage,
                ]);
                throw new ValuationRequestFailed('Failed to get valuation details from BCA.');
            })->json();

        if (empty($valuation)) {
            throw new ValuationRequestFailed('Failed to get valuation details from BCA');
        }

        return new ValuationResponse(
            provider: 'bca',
            registration: $registration,
            mileage: $mileage,
            vehicleType: $vehicleSpecification['vehicleType'] ?? '',
            retailPrice: null,
            averagePrice: $valuation['current']['valuation'],
            cleanPrice: null,
            belowPrice: null,
            make: $vehicleSpecification['make'] ?? '',
            model: $vehicleSpecification['model'] ?? '',
            derivative: $vehicleSpecification['derivative'] ?? '',
            fuelType: $vehicleSpecification['fuelType'] ?? '',
            engineSize: $vehicleSpecification['engineSize'] ?? '',
            bodyType: $vehicleSpecification['bodyType'] ?? '',
            dateOfValuation: now(),
            transmission: $vehicleSpecification['transmissionType'] ?? '',
            registrationDate: Carbon::createFromFormat('Y-m-d\TH:i:s', $vehicleSpecification['registrationDate']),
            colour: $vehicleSpecification['colour'] ?? '',
            data: [
                'vehicleId' => $valuation['vehicleId'] ?? null,
                'correlationId' => $valuation['correlationId'] ?? null,
            ],
            cosmetic_condition: $this->getCosmeticCondition($cosmetic_condition),
        );
    }

    protected function headers(): array
    {
        return [
            'Authorization' => "Bearer " . $this->accessToken(),
        ];
    }

    protected function accessToken(): string
    {
        $cacheKey = 'bca-api-access-token-' . $this->config->apiKey();

        return Cache::remember(
            $cacheKey,
            Carbon::now()->addSeconds(3599),
            fn() => Http::asForm()->post($this->config->getAuthEndpoint(), [
                'client_id' => $this->config->apiKey(),
                'client_secret' => $this->config->apiSecret(),
                'scope' => $this->config->getScope(),
                'grant_type' => 'client_credentials',
            ])
                ->onError(fn(Response $error) => Log::error('BCA API token error', [
                    'endpoint' => $this->config->getAuthEndpoint(),
                    'response' => $error->body(),
                ]))
                ->json('access_token')
        );
    }

    /**
     * @param string $derivativeId
     * @param int $mileage
     * @return string
     */
    public function valuationEndpoint(string $derivativeId, int $mileage): string
    {
        $params = [
            'advertiserId' => $this->config->merchantId(),
            'derivativeId' => $derivativeId,
            'odometerReadingMiles' => $mileage,
        ];

        return $this->endpoint("valuations?" . http_build_query($params));
    }

    /**
     * @param string $registration
     * @return array|null
     * @throws ValuationRequestFailed
     */
    public function getVehicleData(string $registration): ?array
    {
        return Http::withHeaders($this->headers())
            ->get($this->config->getEndpoint($registration) . 'specification')
            ->onError(function (Response $response) use ($registration) {
                Log::warning('BCA Valuation failed to get derivative ID', [
                    'message' => $response->body(),
                    'registration' => $registration,
                ]);
                throw new ValuationRequestFailed('BCA was not able to find vehicle: ' . $response->json('message'));
            })->json();
    }

    public function getVariants(string $registration_number, int $mileage): array
    {
        return collect($this->getVehicleData($registration_number)['vehicleSpecifications'] ?? [])
            ->map(fn($spec) => [
                'variant_id' => $spec['identifier'],
                'name' => $spec['vehicleDescription'],
                'make' => $spec['make'],
                'model' => $spec['model'],
                'colour' => $spec['colour'],
                'body_type' => $spec['bodyType'],
                'fuel_type' => $spec['fuelType'],
            ])
            ->toArray();
    }

    public function sendCustomerData(ValuationCustomerRequest $request, VehicleValuation $valuation): bool
    {
        return true;
    }

    public function confirmValuation(VehicleValuation $valuation): void
    {
        if (empty($this->config->locationId())) {
            return;
        }
        Http::asJson()->withHeaders($this->headers())
            ->post($this->config->getEndpoint($valuation->registration) . 'orderconfirmation', [
                'isDealt' => true,
                'dueInDate' => Carbon::now()->addDay()->format('Y-m-d'),
                'vehicleId' => $valuation->data['vehicleId'] ?? null,
                "manufacturerRetailerCode" => $this->config->locationId(),
                "valuationPrice" => $valuation->adjusted_average_price ?? $valuation->average_price,
                "correlationId" => $valuation->data['correlationId'] ?? null,
            ])
            ->onError(function (Response $response) use ($valuation) {
                Log::warning('BCA Valuation Order confirmation Request failed', [
                    'message' => $response->body(),
                    'status' => $response->status(),
                    'valuation' => $valuation->id,
                ]);
            })->json();
    }

    protected function getCosmeticCondition(?string $cosmetic_condition): string
    {
        if (empty($cosmetic_condition)) {
            return 'Excellent';
        }

        return match (strtolower($cosmetic_condition)) {
            'poor' => 'Poor',
            'below_average' => 'BelowAverage',
            'average' => 'Average',
            'good' => 'Good',
            'excellent' => 'Excellent',
            default => 'Excellent',
        };
    }
}
