<?php

namespace Mtc\VehicleSavings;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Vehicle;

class SavingsCalculatorPeterVardy extends SavingsCalculatorDefault
{
    /**
     * @param Vehicle $vehicle
     * @return bool
     */
    public function VehicleIsEligibleForSavings(Vehicle $vehicle): bool
    {
        // savings not allowed for vehicles at Porsche dealerships
        $ineligible_dealerships = [
            'PA' => 'PA',
            'PP' => 'PP'
        ];

        if (
            empty($vehicle->dealership)
            || array_key_exists($vehicle->dealership['location_stock'], $ineligible_dealerships)
        ) {
            return false;
        }

        return true;
    }

    /**
     * @param Vehicle $vehicle
     * @return Collection
     */
    protected function calculateSavings(Vehicle $vehicle): Collection
    {
        $total = 0;

        // sanitise savings and calculate total
        $savings_sanitised = collect($this->calculateSavingsForVehicle($vehicle))->map(function ($saving) use (&$total) {

            // don't process null savings items
            if ($saving == null) {
                return null;
            }

            if ($saving < $this->savings_floor) {
                $saving = $this->savings_floor;
            }

            $total += $saving;

            return $saving;
        });

        $savings_sanitised->put('saving_total', $total);

        return $savings_sanitised;
    }

    private function calculateSavingsForVehicle(Vehicle $vehicle): Collection
    {
        $savings = collect([
            'saving_original_price' => null,
            'saving_rrp' => null,
            'saving_finance_rate' => null,
            'saving_finance_contribution' => null,
        ]);

        if ($this->isBestBuy($vehicle)) {
            $savings = $savings->merge($this->getBestBuySavings($vehicle));
        } elseif ($this->isPremiumMake($vehicle)) {
            $savings = $savings->merge($this->getPremiumSavings($vehicle));
        } elseif ($this->isValueRange($vehicle)) {
            $savings = $savings->merge($this->getValueRangeSavings($vehicle));
        } else {
            $savings = $savings->merge($this->getDefaultSavings($vehicle));
        }

        return $savings;
    }

    private function isBestBuy(Vehicle $vehicle): bool
    {
        // check if age is 12 months or under
        if (!empty($vehicle->first_registration_date)) {

            $reg_date = Carbon::createFromFormat('Y-m-d H:i:s', $vehicle->first_registration_date);

            if (!empty($reg_date) && $reg_date->diffInMonths(Carbon::now()) <= 12) {
                return true;
            }
        }

        return false;
    }

    private function isPremiumMake(Vehicle $vehicle): bool
    {
        return array_key_exists(strtoupper($vehicle->make?->name), $this->getPremiumMakes());
    }

    private function getPremiumMakes()
    {
        return [
            'AUDI' => 'AUDI',
            'BMW' => 'BMW',
            'JAGUAR' => 'JAGUAR',
            'LAND ROVER' => 'LAND ROVER',
            'MERCEDES-BENZ' => 'MERCEDES-BENZ',
            'MERCEDES BENZ' => 'MERCEDES BENZ',
            'MERCEDES' => 'MERCEDES',
            'LEXUS' => 'LEXUS',
        ];
    }

    private function isValueRange(Vehicle $vehicle): bool
    {
        return ($vehicle->price < 10000);
    }

    /**
     * "Best Buy" savings
     * cash saving
     * finance rate saving
     *
     * @param Vehicle $vehicle
     * @return Collection
     */
    private function getBestBuySavings(Vehicle $vehicle): Collection
    {
        return $this->getPriceSaving($vehicle)
            ->merge($this->getFinanceRateSaving($vehicle));
    }

    /**
     * "premium" vehicle savings
     * cash saving
     * finance rate saving
     * finance cashback saving
     *
     * @param Vehicle $vehicle
     * @return Collection
     */
    private function getPremiumSavings(Vehicle $vehicle): Collection
    {
        return $this->getPriceSaving($vehicle)
            ->merge($this->getFinanceRateSaving($vehicle))
            ->merge($this->getFinanceContribution($vehicle));
    }

    /**
     * "Value Range" savings
     * cash saving
     *
     * @param Vehicle $vehicle
     * @return Collection
     */
    private function getValueRangeSavings(Vehicle $vehicle): Collection
    {
        return $this->getPriceSaving($vehicle);
    }

    /**
     * default savings for "core" vehicles
     * cash saving
     * finance rate saving
     * finance cashback saving
     *
     * @param Vehicle $vehicle
     * @return Collection
     */
    private function getDefaultSavings(Vehicle $vehicle): Collection
    {
        return $this->getPriceSaving($vehicle)
            ->merge($this->getFinanceRateSaving($vehicle))
            ->merge($this->getFinanceContribution($vehicle));
    }

    private function getFinanceRateSaving(Vehicle $vehicle): Collection
    {
        $saving = collect([
            'saving_finance_rate' => 0
        ]);

        if (Settings::get('vehicle-savings-enabled-finance-saving') === false) {
            return $saving;
        }

        $finance_example = $vehicle->defaultFinanceExample;

        if (empty($finance_example)) {
            return $saving;
        }

        $saving_pounds_per_thousand = Settings::get('vehicle-savings-pv-finance-saving-pounds-per-thousand', 0);
        $saving->put('saving_finance_rate', $saving_pounds_per_thousand * (floor($finance_example->total_credit_amount / 1000)));
        return $saving;
    }

    /**
     * Finance Contribution, AKA Cashback saving
     */
    private function getFinanceContribution(Vehicle $vehicle): Collection
    {
        $saving = collect([
            'saving_finance_contribution' => 0
        ]);

        if (Settings::get('vehicle-savings-enabled-finance-contribution') === false) {
            return $saving;
        }

        $finance_example = $vehicle->defaultFinanceExample;

        if (empty($finance_example)) {
            return $saving;
        }

        if ($this->isEligibleForCashbackSaving($vehicle, $finance_example->total_credit_amount)) {
            $saving->put('saving_finance_contribution',  $this->calculateFinanceContribution($vehicle, $finance_example->total_credit_amount));
        }

        return $saving;
    }

    private function isEligibleForCashbackSaving(Vehicle $vehicle, $amount_of_credit = 0)
    {
        if ($this->vehicleHasCarzOrVanzDealership($vehicle)
            && $vehicle->type === VehicleType::CAR->value
            && $amount_of_credit > 10000) {
            return true;
        }

        return false;
    }

    /**
     * Note that this is the saving calculation to be used
     * when the client describes an x% APR finance offer.
     *
     * finance contribution/cashback
     *
     * @param $amount_to_finance
     * @return float|int
     */
    private function calculateFinanceContribution(Vehicle $vehicle, $amount_to_finance = 0)
    {
        $finance_saving = 0;

        $saving_percentage_combustion = $this->getFinanceContributionPercentageICEVehicle();
        $saving_percentage_electric = $this->getFinanceContributionPercentageElectricVehicle();
        $saving_percentage = $saving_percentage_combustion;
        $saving_cap_combustion = Settings::get('vehicle-savings-pv-finance-contribution-max-ice');
        $saving_cap_electric = Settings::get('vehicle-savings-pv-finance-contribution-max-electric');
        $saving_cap = $saving_cap_combustion;

        if (strtoupper($vehicle->fuelType?->name ?? '') == 'ELECTRIC') {
            $saving_percentage = $saving_percentage_electric;
            $saving_cap = $saving_cap_electric;
        }

        // saving is % of amount to finance
        if($amount_to_finance > 0){
            $finance_saving = round($amount_to_finance * $saving_percentage);
        }

        // T150631 Cap "FDA" saving
        if ($finance_saving > $saving_cap) {
            $finance_saving = $saving_cap;
        }

        return $finance_saving;
    }

    private function getPriceSaving(Vehicle $vehicle): Collection
    {
        $savings = collect([]);
        $vehicle_price = $this->getVatAwareVehiclePrice($vehicle);
        $saving_original_price = $vehicle->previous_price - $vehicle_price;
        $saving_rrp = $vehicle->rrp_price - $vehicle_price;

        if (!$this->priceSavingIsAllowed($vehicle, $saving_original_price)) {
            $saving_original_price = null;
        }

        if (!$this->priceSavingIsAllowed($vehicle, $saving_rrp, true)) {
            $saving_rrp = null;
        }

        // prefer saving from original price, falling back to saving from RRP
        if ($saving_original_price > 0) {
            $saving_rrp = null;
        } elseif ($saving_rrp > 0) {
            $saving_original_price = null;
        }

        $savings->put('saving_original_price', $saving_original_price);
        $savings->put('saving_rrp', $saving_rrp);

        return $savings;
    }

    private function priceSavingIsAllowed(Vehicle $vehicle, float $saving, bool $is_rrp_saving = false): bool
    {
        if ($saving <= 0
            || (
                (!Settings::get('vehicle-savings-enabled-previous-price') && !$is_rrp_saving)
                || (!Settings::get('vehicle-savings-enabled-rrp') && $is_rrp_saving)
            )) {
            return false;
        }

        if ($this->vehicleHasCarzOrVanzDealership($vehicle)) {
            if ($saving >= ($is_rrp_saving ? 50 : 100)) {
                return true;
            } else {
                return false;
            }
        }

        if ($this->vehicleHasPremiumDealership($vehicle)) {
            if ($vehicle->price < 20000 && $saving >= 100) {
                return true;
            }

            if ($vehicle->price >= 20000 && $saving >= 200) {
                return true;
            }
        }

        return false;
    }

    private function vehicleHasCarzOrVanzDealership(Vehicle $vehicle): bool
    {
        if (
            $vehicle->dealership
            && array_key_exists($vehicle->dealership['location_stock'], $this->getCarzAndVanzDealerCodes())
        ) {
            return true;
        }

        return false;
    }

    private function getCarzAndVanzDealerCodes()
    {
        $carz_dealerships = [
            'CD' => 'CD', // carz Dundee
            'ML' => 'ML', // carz Bruntingthrope
            'UG' => 'UG', // carz Hillington Glasgow
            'VA' => 'VA', // carz Aberdeen
            'VE' => 'VE', // carz Edinburgh
            'VF' => 'VF', // carz Dalgety Bay
            'VK' => 'VK', // carz Kirkcaldy
            'VM' => 'VM', // carz Motherwell
            'VP' => 'VP', // carz Perth
        ];

        // all Carz Dealerships are also Vanz dealerships
        $vanz_dealerships = [];

        foreach ($carz_dealerships as $carz_dealer) {
            $key = $carz_dealer . '-VANZ';
            $vanz_dealerships[$key] = $key;
        }

        return array_merge($carz_dealerships, $vanz_dealerships);
    }

    private function vehicleHasPremiumDealership(Vehicle $vehicle): bool
    {
        if (
            $vehicle->dealership
            && in_array($vehicle->dealership['location_stock'], $this->getPremiumDealerships())
        ) {
            return true;
        }

        return false;
    }

    private function getPremiumDealerships()
    {
        return [
            'BE', // BMW Edinburgh
            'JA', // Jaguar Aberdeen
            'LA', // Land Rover Aberdeen
            'ME', // MINI Edinburgh
            'HC', // Prestige & Peformance (the franchise formerly known as Heritage)
        ];
    }

    /**
     * Return the vehicle price for VAT eligible vehicles.
     * Return the vehicle price minus VAT for VAT excluded vehicles.
     *
     * @param Vehicle $vehicle
     * @return float
     */
    private function getVatAwareVehiclePrice(Vehicle $vehicle): float
    {
        if (empty($vehicle->price)) {
            return 0;
        }

        if ($vehicle->is_vat_applicable) {
            return $vehicle->price;
        }

        return $vehicle->price / 1.2;
    }

    private function getFinanceContributionPercentageICEVehicle()
    {
        $int_value = Settings::get('vehicle-savings-pv-finance-contribution-percent-ice', 0);
        return (float) ($int_value / 100);
    }

    private function getFinanceContributionPercentageElectricVehicle()
    {
        $int_value = Settings::get('vehicle-savings-pv-finance-contribution-percent-electric', 0);
        return (float) ($int_value / 100);
    }
}
