<?php

namespace Mtc\VehicleReservations\Jobs;

use App\Facades\Settings;
use App\Services\EnquiryMaxApi;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Mtc\VehicleReservations\Mail\NewReservationTeamMail;
use Mtc\VehicleReservations\Mail\ReservationCancelledCustomerMail;
use Mtc\VehicleReservations\Mail\ReservationCancelledTeamMail;
use Mtc\VehicleReservations\Reservation;

class ProcessIncomingReservationJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected Reservation $reservation)
    {
        $this->onQueue('emails');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        if ($this->process()) {
            $this->reservation->refresh()->update([
                'ingested_at' => Carbon::now(),
            ]);
        }
    }

    private function process(): bool
    {
        if (Settings::get('crm-enquiry-max-send-reservations')) {
            try {
                App::make(EnquiryMaxApi::class)->sendReservation($this->reservation);
            } catch (\Exception $exception) {
                Log::error(tenant('id') . ' failed to export reservation to enquiryMax', [$exception->getMessage()]);
            }
        }
        return $this->sendConfirmation();
    }

    private function sendConfirmation(): bool
    {
        if (!empty($this->reservation->confirmation_sent_at)) {
            return true;
        }

        try {

            $this->notifyCustomer($this->reservation);
            $this->notifyAdminTeam($this->reservation);

            return true;
        } catch (\Exception $exception) {
            Log::warning($exception->getMessage());
            $this->reservation->refresh()
                ->update(['attempts' => $this->reservation->attempts + 1]);
            return false;
        }
    }

    private function notifyCustomer(Reservation $reservation): void
    {
        if (empty($reservation->email)) {
            return;
        }
        $confirmation_class = config('reservations.reservation_confirmation_mail');
        $mailable = match ($reservation->status) {
            'confirmed' => new $confirmation_class($reservation),
            'cancelled' => new ReservationCancelledCustomerMail($reservation),
        };
        Mail::to($reservation->email)->send($mailable);
    }

    private function notifyAdminTeam(Reservation $reservation): void
    {
        $confirmation_class = config('reservations.reservation_confirmation_mail');
        $mailable = match ($reservation->status) {
            'confirmed' => new $confirmation_class($reservation),
            'cancelled' => new ReservationCancelledTeamMail($reservation),
        };
        $team_emails = $this->getTeamEmails();
        if (!empty($team_emails)) {
            Mail::to($team_emails)->send($mailable);
        }
    }

    private function getTeamEmails(): array
    {
        return array_filter(
            explode(',', Settings::get('reservations-admin-team-emails')),
            fn($email) => $email && filter_var($email, FILTER_VALIDATE_EMAIL)
        );
    }
}
