<?php

namespace Mtc\VehicleReservations;

use App\Facades\Settings;
use App\IntegrationRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\InvoiceFactory;
use Mtc\MercuryDataModels\Contracts\FacilitatesReservations;
use Mtc\MercuryDataModels\Vehicle;

class ReservationRepository
{
    public function getVehicleAction(Vehicle $vehicle): array
    {
        return [
            'type' => 'link',
            'vehicle' => $vehicle->slug,
        ];
    }

    public function create(array $data, Vehicle $vehicle): Model
    {
        if (empty($data['name']) && !empty($data['first_name'])) {
            $data['name'] = $data['first_name'] . ' ' . $data['last_name'];
        }
        $data['status'] = 'pending';
        $data['reference'] = strtoupper(Str::random(12));
        $data['data'] = $this->getReservationData($vehicle, $data);
        return Reservation::query()->create($data);
    }

    public function createInvoice(Reservation $reservation): InvoiceRepositoryContract
    {
        return (new InvoiceFactory())->create($reservation);
    }

    public function updateDetails(Reservation $reservation, array $details): Reservation
    {
        $reservation->fill($details);
        $this->updateInvoice($reservation);
        $reservation->save();
        return $reservation;
    }

    public function addEmailWhenRequired(Reservation $reservation, ?string $email = null): void
    {
        if (empty($reservation->email) && $email) {
            $reservation->update([
                'email' => $email
            ]);
        }
    }

    public function markConfirmed(Reservation $reservation, float $amount_paid, string $reference): bool
    {
        try {
            $reservation->update([
                'status' => 'confirmed',
                'confirmed_at' => Carbon::now(),
                'payment_reference' => $reference,
                'amount' => $amount_paid,
            ]);

            if ($this->shouldMarkVehicleReservedOnConfirmation()) {
                $reservation->vehicle->update([
                    'is_reserved' => true,
                    'reserved_at' => Carbon::now(),
                ]);
            }
            return true;
        } catch (\Exception $exception) {
            Log::error('Unable to mark reservation as confirmed', [
                $reservation,
                $exception->getMessage(),
                $exception->getTrace(),
            ]);
            return false;
        }
    }

    public function cancel(Reservation $reservation): bool
    {
        if ($this->shouldRefund($reservation) && $this->canRefund($reservation)) {
            return $this->refund($reservation);
        }

        $reservation->update(['status' => 'cancelled']);
        if ($this->vehicleHasNoReservation($reservation->vehicle)) {
            $reservation->vehicle->update(['is_reserved' => false]);
        }

        return false;
    }

    public function getPaymentConfig(Reservation $reservation, InvoiceRepositoryContract $invoice_repository): array
    {
        return match ($this->getPaymentProvider()) {
            'stripe' => $this->stripeConfig($reservation, $invoice_repository),
            'vyne' => $this->vyneConfig($reservation, $invoice_repository),
            default => [],
        };
    }

    /**
     * @param Vehicle $vehicle
     * @return bool
     */
    public function vehicleIsEligibleForReservation(Vehicle $vehicle): bool
    {
        if ($vehicle->is_reserved || $vehicle->is_sold) {
            return false;
        }

        $providers = (new IntegrationRepository())
            ->getEnabledForType('stock-reservation')
            ->filter(fn($integration) => !empty($integration['class']))
            ->map(fn($integration) => App::make($integration['class']))
            ->filter(fn($integration) => $integration instanceof FacilitatesReservations);

        if ($providers->isEmpty()) {
            return true;
        }

        return $providers->count() == $providers
            ->filter(fn(FacilitatesReservations $provider) => $provider->vehicleAvailableForReservation($vehicle))
            ->count();
    }

    /**
     * @param Reservation $reservation
     * @return void
     */
    public function setVehicleReservedOnThirdPartySystems(Reservation $reservation): void
    {
        (new IntegrationRepository())
            ->getEnabledForType('stock-reservation')
            ->filter(fn($integration) => !empty($integration['class']))
            ->map(fn($integration) => App::make($integration['class']))
            ->filter(fn($integration) => $integration instanceof FacilitatesReservations)
            ->each(fn(FacilitatesReservations $provider) => $provider->setReserved($reservation->vehicle));
    }

    protected function getPaymentProvider(): string
    {
        // todo: dynamic
        return 'stripe';
    }

    protected function stripeConfig(Reservation $reservation, InvoiceRepositoryContract $invoice_repository): array
    {
        return [
            'provider' => 'stripe',
            'invoice_id' => $invoice_repository->getId(),
            'amount' => $invoice_repository->getOutstandingAmount() * 100,
            'country' => Settings::get('app-details-country'),
            'currency' => $invoice_repository->getCurrency(),
            'item_name' => 'Reservation fee for ' . $invoice_repository->getModel()->items->pluck('name')->implode(','),
            'stripe_public_key' => App::make(config('stripe.config'))->publicKey(),
            'customer' => null,
        ];
    }


    protected function vyneConfig(Reservation $reservation, InvoiceRepositoryContract $invoice_repository): array
    {
        /** @var Vyne $vyne */
        $vyne = App::make(Vyne::class);
        return [
            'provider' => 'vyne',
            'payment_details' => $vyne->initializePayment($invoice_repository),
            'invoice_id' => $invoice_repository->getId(),
            'amount' => $invoice_repository->getOutstandingAmount() * 100,
            'country' => Settings::get('app-details-country'),
            'currency' => $invoice_repository->getCurrency(),
            'item_name' => 'Reservation fee for ' . $invoice_repository->getModel()->items->pluck('name')->implode(','),
            'merchant_id' => Settings::get('sales-vyne-merchant-id'),
            'customer' => null,
        ];
    }

    /**
     * @param array $data
     * @param Vehicle|null $vehicle
     * @return array
     */
    protected function getReservationData(Vehicle $vehicle, array $input): array
    {
        $return_data = [];

        $return_data['vehicle']['description'] = implode(' ', [
            $vehicle->registration_number,
            $vehicle->make?->name ?? '',
            $vehicle->model?->name ?? '',
            $vehicle->derivative,
            $vehicle->price
        ]);
        $return_data['meta'] = [
            'utm_campaign' => $input['utm_campaign'] ?? null,
            'utm_content' => $input['utm_content'] ?? null,
            'utm_medium' => $input['utm_medium'] ?? null,
            'utm_source' => $input['utm_source'] ?? null,
            'utm_term' => $input['utm_term'] ?? null,
        ];

        return $return_data;
    }

    private function shouldRefund(Reservation $reservation): bool
    {
        return $reservation->status == 'confirmed';
    }

    private function canRefund(Reservation $reservation): bool
    {
        return $reservation->invoice->isRefundable();
    }

    private function refund(Reservation $reservation): bool
    {
        return $invoice_repository->refund($reservation->invoice);
    }

    private function vehicleHasNoReservation(Vehicle $vehicle): bool
    {
        return $vehicle->reservations()
            ->whereNotIn('status', ['pending', 'cancelled'])
            ->exists();
    }


    private function shouldMarkVehicleReservedOnConfirmation(): bool
    {
        return Settings::get('mark-vehicle-reserved-on-confirmation', true);
    }

    private function updateInvoice(Reservation $reservation): void
    {
        if ($reservation->isDirty('email')) {
            $reservation->invoice->update([
                'email' => $reservation->email
            ]);
        }
    }
}
