<?php

namespace Mtc\VehicleLookup\Drivers;

use App\Modules\Lookup\Contracts\VehicleLookupData;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\VehicleLookup\Config\HPIConfig;
use Mtc\VehicleLookup\Contracts\VehicleLookupDriver;
use Mtc\VehicleLookup\VehicleLookupResponse;
use Exception;

class HPI implements VehicleLookupDriver
{
    public function __construct(protected readonly HPIConfig $config)
    {
        //
    }

    public function lookup(string $registration_number, int $mileage): VehicleLookupResponse
    {
        return $this->parseResponse(
            $registration_number,
            $mileage,
            $this->getVehicleData($registration_number)
        );
    }

    private function parseResponse(string $registration_number, $mileage, $vehicleData): VehicleLookupResponse
    {
        if (is_null($vehicleData)) {
            return new VehicleLookupResponse($registration_number, $mileage);
        }

        return new VehicleLookupResponse(
            $registration_number,
            $mileage,
            make: data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.Make.Description'),
            model: data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.Model.Description'),
            derivative: data_get($vehicleData, 'Asset.PrimaryAssetData.SMMT.Model.Description'),
            manufacture_year: (int)data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.KeyDates.Manufactured.Year'),
            colour: data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.Body.Colour.Current.Description'),
            body_type: data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.Body.Description'),
            fuel_type: data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.Engine.Fuel.Description'),
            transmission: data_get($vehicleData, 'Asset.PrimaryAssetData.SMMT.Transmission.Description'),
            registration_date: data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.KeyDates.FirstRegistered.Date'),
            co2: data_get($vehicleData, 'Asset.PrimaryAssetData.AdditionalInformation.CO2Rating'),
            engine_capacity_cc: data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.Engine.Size'),
            body_style: data_get($vehicleData, 'Asset.PrimaryAssetData.SMMT.Body.Description'),
            drivetrain: data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.Body.WheelPlan.Description'),
            vin: data_get($vehicleData, 'Asset.AssetIdentification.Vin'),
            doors: (int)data_get($vehicleData, 'Asset.PrimaryAssetData.DVLA.Body.Doors'),
            seats: (int)data_get($vehicleData, 'Asset.SupplementaryAssetData.SmmtExtra.Main.NumberOfSeats'),
            previous_owner_count: (int)data_get(
                $vehicleData,
                'Asset.PrimaryAssetData.DVLA.Keepers.TotalPreviousKeepers'
            ),
        );
    }

    public function findDerivatives(VehicleMake $make, VehicleModel $model): array
    {
        return [];
    }

    public function findVariants(VehicleMake $make, VehicleModel $model, string $derivative = ''): array
    {
        return [];
    }

    public function getTechnicalData(string $variant): VehicleLookupData
    {
        return new VehicleLookupData();
    }

    protected function getVehicleData($registration_number): ?\SimpleXMLElement
    {
        try {
            $response = Cache::get('HPI_stored_response_' . tenant()->id . '_' . $registration_number);
            if (!$response) {
                $response = $this->call($this->getVehicleRequest($registration_number))->body();
            }

            $xml = simplexml_load_string($response);

            return $xml
                ->children('http://schemas.xmlsoap.org/soap/envelope/')
                ->Body
                ->children('http://webservices.hpi.co.uk/SupplementaryEnquiryV1')
                ->EnquiryResponse
                ->RequestResults;
        } catch (Exception $exception) {
            Log::error(__CLASS__ . '::' . __FUNCTION__ . '() ' . $exception->getMessage());
            return null;
        }
    }

    protected function call(array $request_data)
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(20)
            ->withBody($this->getEnvelope($request_data), 'text/xml')
            ->post($this->config->endpoint());

        if ($response->failed()) {
            Log::error('HPI call failed', [
                'endpoint' => $this->config->endpoint(),
                'request_data' => $request_data,
                'status_code' => $response->status(),
                'response' => $response->body(),
            ]);
            throw new Exception('Failed to perform call to ' . $this->config->endpoint());
        }

        return $response;
    }

    protected function getVehicleRequest($registration_number): array
    {
        return [
            'subscriber_details' => [
                'customer_code' => $this->config->customerCode(),
                'initials' => $this->config->initials(),
                'password' => $this->config->password(),
            ],
            'asset' => [
                'vrm' => $registration_number,
                'reference' => '',
            ],
            'primary_product' => $this->config->primaryProductCode(),
            'supplementary_product' => $this->config->supplementaryProductCode(),
        ];
    }

    protected function getHeaders(): array
    {
        return [
            'Content-Type' => 'text/xml; charset=utf-8',
            'SOAPAction' => 'http://webservices.hpi.co.uk/SupplementaryEnquiryV1/enquire',
        ];
    }

    protected function getEnvelope(array $request_data): string
    {
        return view('lookup::hpi_enquiry_lookup_request', ['data' => $request_data])->render();
    }
}
