<?php

namespace Mtc\VehicleLookup\Drivers;

use App\Master\Services\AutoTraderTaxonomyService;
use App\Modules\Lookup\Contracts\VehicleLookupData;
use App\VehicleSpec\Contracts\VehicleStandardEquipmentItem;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\VehicleLookup\Contracts\VehicleLookupDriver;
use Mtc\VehicleLookup\VehicleLookupResponse;
use Mtc\VehicleValuation\AutoTrader as ValuationAutoTrader;
use Mtc\VehicleValuation\Config\AutoTraderConfig;

class AutoTrader extends ValuationAutoTrader implements VehicleLookupDriver
{
    protected AutoTraderTaxonomyService $service;

    public function __construct(protected AutoTraderConfig $config)
    {
        $this->service = app(AutoTraderTaxonomyService::class);
        parent::__construct($this->config);
    }

    public function lookup(string $registration_number, int $mileage): VehicleLookupResponse
    {
        $flags = [
            'history' => 'true',
            'features' => 'true',
            'chargeTimes' => 'true',
        ];
        return $this->parseResponse(
            $registration_number,
            $mileage,
            $this->getVehicleData($registration_number, null, $flags)
        );
    }

    private function parseResponse(string $registration_number, $mileage, array $vehicle_data): VehicleLookupResponse
    {
        $vehicle = $vehicle_data['vehicle'];
        $year = isset($vehicle['firstRegistrationDate'])
            ? Carbon::parse($vehicle['firstRegistrationDate'])->format('y')
            : null;
        $charging_data = [];
        if (!empty($vehicle_data['chargeTimes'])) {
            $charging_data = $this->parseChargeTimeData($vehicle_data);
        }
        return new VehicleLookupResponse(
            $registration_number,
            $mileage,
            make: $vehicle['make'] ?? null,
            model: $vehicle['model'] ?? null,
            derivative: $vehicle['derivative'] ?? null,
            manufacture_year: $vehicle_data['history']['yearOfManufacture'] ?? $year,
            colour: $vehicle['colour'] ?? null,
            body_type: $vehicle['bodyType'] ?? null,
            fuel_type: $vehicle['fuelType'] ?? null,
            transmission: $vehicle['transmissionType'] ?? null,
            registration_date: $vehicle['firstRegistrationDate'] ?? null,
            mpg: $vehicle['fuelEconomyWLTPCombinedMPG'] ?? null,
            co2: $vehicle['co2EmissionGPKM'] ?? null,
            engine_capacity_cc: $vehicle['engineCapacityCC'] ?? null,
            body_style: $vehicle['bodyType'] ?? null,
            drivetrain: $vehicle['drivetrain'] ?? null,
            vehicle_type: $vehicle['vehicleType'] ?? null,
            vin: $vehicle['vin'],
            doors: $vehicle['doors'] ?? null,
            seats: $vehicle['seats'] ?? null,
            previous_owner_count: $vehicle_data['history']['previousOwners'] ?? null,
            features: collect($vehicle_data['features'] ?? [])
                ->filter(fn($row) =>  !empty($row['factoryFitted'])),
            standard_equipment: collect($vehicle_data['features'] ?? [])
                ->map(fn($row) => new VehicleStandardEquipmentItem('', $row['name'], $row['category'], $row['type'])),
            battery_capacity_kwh: $charging_data['battery_capacity_kwh'] ?? null,
            battery_charge_time: $charging_data['battery_charge_time'] ?? null,
            battery_quick_charge_time: $charging_data['battery_quick_charge_time'] ?? null,
            battery_quick_charge_start_level: $charging_data['battery_quick_charge_start_level'] ?? null,
            battery_quick_charge_level: $charging_data['battery_quick_charge_level'] ?? null,
            plug_type: $charging_data['plug_type'] ?? null,
            rapid_charge_plug_type: $charging_data['rapid_charge_plug_type'] ?? null,
            battery_slow_charge_description: $charging_data['battery_slow_charge_description'] ?? null,
            battery_quick_charge_description: $charging_data['battery_quick_charge_description'] ?? null,
        );
    }

    public function findDerivatives(VehicleMake $make, VehicleModel $model): array
    {
        return array_map(fn($generation) => [
            'id' => $generation['generationId'],
            'name' => $generation['name'],
        ], Http::withHeaders($this->headers())
            ->get($this->endpoint('taxonomy/generations?' . http_build_query([
                    'modelId' => $model->autotrader_id,
                    'advertiserId' => $this->config->merchantId(),
                ])))
            ->json('generations') ?? []);
    }

    public function findVariants(VehicleMake $make, VehicleModel $model, string $generation = ''): array
    {
        return array_map(fn($generation) => [
            'id' => $generation['derivativeId'],
            'name' => $generation['name'],
        ], Http::withHeaders($this->headers())
            ->get($this->endpoint('taxonomy/derivatives?' . http_build_query([
                    'generationId' => $generation,
                    'advertiserId' => $this->config->merchantId(),
                ])))
            ->json('derivatives') ?? []);
    }

    /**
     * Fetches the technical data for a vehicle.
     *
     * This function retrieves the technical data for a specific vehicle variant
     * from the AutoTrader API.
     *
     * @link https://developers.autotrader.co.uk/api#technical-data AutoTrader Technical Data Documentation
     *
     * @param string $variant The variant identifier for the vehicle.
     * @return VehicleLookupData The technical data for the vehicle.
     */
    public function getTechnicalData(string $variant): VehicleLookupData
    {
        $response = Http::withHeaders($this->headers())
            ->get($this->endpoint('taxonomy/derivatives/' . $variant . '?' . http_build_query([
                    'advertiserId' => $this->config->merchantId(),
                ])))
            ->json();

        return new VehicleLookupData(
            make: $response['make'] ?? null,
            model: $response['model'] ?? null,
            generation: $response['generation'] ?? null,
            derivative: $response['name'] ?? null,
            trim: $response['trim'] ?? null,
            body_type: $response['bodyType'] ?? null,
            vehicle_type: $response['vehicleType'] ?? null,
            fuel_type: $response['fuelType'] ?? null,
            transmission: $response['transmissionType'] ?? null,
            drivetrain: $response['drivetrain'] ?? null,
            seats: $response['seats'] ?? null,
            doors: $response['doors'] ?? null,
            drive_type: $response['driveType'] ?? null,
            engine_capacity_cc: $response['engineCapacityCC'] ?? null,
        );
    }

    private function parseChargeTimeData(array $vehicle_data): array
    {
        $charge_times = collect($vehicle_data['chargeTimes'] ?? []);
        $fastest = $charge_times->sortBy('durationMinutes')->first();
        $slowest = $charge_times->sortByDesc('durationMinutes')->first();

        return [
            'battery_capacity_kwh' => $vehicle_data['vehicle']['batteryCapacityKWH'] ?? null,
            'battery_charge_time' => $slowest['durationMinutes'] ?? null ,
            'battery_slow_charge_description' => implode(' - ', array_filter([
                $slowest['chargerSupplyType'] ?? null,
                $slowest['chargerDescription'] ?? null,
            ])),
            'battery_quick_charge_time' => $fastest['durationMinutes'] ?? null,
            'battery_quick_charge_start_level' => $fastest['startBatteryPercentage'] ?? null,
            'battery_quick_charge_level' => $fastest['endBatteryPercentage'] ?? null,
            'battery_quick_charge_description' => implode(' - ', array_filter([
                $fastest['chargerSupplyType'] ?? null,
                $fastest['chargerDescription'] ?? null,
            ])),
            'plug_type' => $slowest['connectorType'] ?? null,
            'rapid_charge_plug_type' => $fastest['connectorType'] ?? null,
        ];
    }
}
