<?php

namespace Mtc\Taxonomies\Contracts;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Str;
use Kalnoy\Nestedset\NodeTrait;
use Mtc\CustomFields\Traits\HasCustomFields;
use Mtc\Taxonomies\Http\Presenters\TaxonomyUrlPresenter;

/**
 * Class Taxonomy
 *
 * @package Mtc\Taxonomies
 */
abstract class TaxonomyModel extends Model
{
    use NodeTrait;
    use HasCustomFields;

    protected $table = 'taxonomies';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'description',
        'image',
        'active',
        'tree_structure',
        'max_depth',
    ];

    /**
     * @var array
     */
    protected $casts = [
        'tree_structure' => 'boolean',
        'active' => 'boolean',
        'custom_fields' => 'array',
    ];

    /**
     * @var array
     */
    protected $appends = [
        'url',
    ];

    /**
     * Boot model
     */
    public static function boot()
    {
        parent::boot();

        static::creating(function (self $taxonomy) {
            $taxonomy->ensureUniqueSlug();
        });

        static::addGlobalScope(function ($query) {
            return $query->orderBy('_lft');
        });
    }

    /**
     * URL builder
     *
     * @return TaxonomyUrlPresenter
     */
    public function getUrlAttribute()
    {
        return new TaxonomyUrlPresenter($this);
    }

    /**
     * Make sure that slug for the taxonomy is unique
     */
    protected function ensureUniqueSlug()
    {
        $i = 0;
        $slug_base = $slug = Str::slug($this->name);
        while (self::query()->where('slug', $slug)->exists()) {
            $i++;
            $slug = "{$slug_base}-{$i}";
        }
        $this->slug = $slug;
    }

    /**
     * Find model by slug with fallback of finding by id
     *
     * @param string|int $slug
     * @return Model
     */
    public function findFromSlug($slug): self
    {
        return self::query()
            ->where('slug', $slug)
            ->orWhere('id', $slug)
            ->firstOrFail();
    }

    /**
     * Get the root level taxonomy of this type
     * @return mixed
     */
    public function getRoot()
    {
        return self::query()
            ->whereIsRoot()
            ->where('_lft', '<=', $this->_lft)
            ->where('_rgt', '>=', $this->_rgt)
            ->first();
    }

    public static function ofType($slug): Builder
    {
        // TODO: this probably isn't nesting safe
        return self::query()
            ->whereHas('parent', function ($parent_query) use ($slug) {
                $parent_query->where('slug', $slug);
            });
    }
}
