<?php

namespace Mtc\Taxonomies\Console\Commands;

use Illuminate\Console\GeneratorCommand;
use Illuminate\Support\Str;
use Mtc\Taxonomies\Facades\Taxonomies;
use Symfony\Component\Console\Input\InputOption;

class MakeTaxonomyCommand extends GeneratorCommand
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'make:taxonomy';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create a new Taxonomy model class';

    /**
     * The type of class being generated.
     *
     * @var string
     */
    protected $type = 'Model';

    public function handle()
    {
        if (parent::handle() === false) {
            return;
        }

        if ($this->taxonomyExists() === false) {
            $this->createDatabaseEntry();
            $this->info('Taxonomy DB entry created');
            return;
        }

        $this->info('Taxonomy DB already exists');
    }

    /**
     * Check if taxonomy db entry exists
     *
     * @return bool
     */
    private function taxonomyExists(): bool
    {
        return Taxonomies::exists($this->getSlug());
    }

    private function createDatabaseEntry()
    {
        return Taxonomies::create($this->getTaxonomyName(), $this->getSlug(), [
            'tree_structure' => $this->option('tree')
        ]);
    }

    /**
     * Build the class details
     *
     * @param string $name
     * @return string
     * @throws \Illuminate\Contracts\Filesystem\FileNotFoundException
     */
    protected function buildClass($name)
    {
        $stub = $this->files->get($this->getStub());

        return $this->replaceNamespace($stub, $name)
            ->replaceSlug($stub)
            ->replaceClass($stub, $name);
    }

    /**
     * Replace the slug for the taxonomy
     *
     * @param $stub
     * @return $this
     */
    protected function replaceSlug(&$stub)
    {
        $stub = str_replace('{{ slug }}', $this->getSlug(), $stub);
        return $this;
    }

    private function getTaxonomyName()
    {
        $words = str_replace('-', ' ', Str::kebab($this->argument('name')));
        return Str::plural(ucwords($words));
    }

    /**
     * Generate slug for the taxonomy
     *
     * @return string
     */
    protected function getSlug(): string
    {
        return Str::slug($this->argument('name'));
    }

    /**
     * Get the stub file for the generator.
     *
     * @return string
     */
    protected function getStub()
    {
        return $this->resolveStubPath('/stubs/taxonomy.stub');
    }

    /**
     * Resolve the fully-qualified path to the stub.
     *
     * @param string $stub
     * @return string
     */
    protected function resolveStubPath($stub)
    {
        return file_exists($customPath = $this->laravel->basePath(trim($stub, '/')))
            ? $customPath
            : __DIR__ . $stub;
    }

    /**
     * Get the default namespace for the class.
     *
     * @param string $rootNamespace
     * @return string
     */
    protected function getDefaultNamespace($rootNamespace)
    {
        return is_dir(app_path('Models')) ? $rootNamespace . '\\Models' : $rootNamespace;
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        return [
            ['tree', 't', InputOption::VALUE_NONE, 'When generating a Taxonomy DB entry create it as a tree structure'],
        ];
    }
}
