<?php

namespace Mtc\Stripe\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\URL;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Checkout\Contracts\InvoiceFactoryContract;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableFactoryContract;
use Mtc\Foundation\Address;
use Mtc\Stripe\Contracts\AbstractStripeConfig;
use Mtc\Stripe\Contracts\StripePaymentRequestBasketFillInterface;
use Mtc\Stripe\Http\Requests\SetAddress;
use Mtc\Stripe\Stripe;
use Stripe\PaymentIntent;

/**
 * Class StripeController
 *
 * @package Mtc\Stripe\Http\Controllers
 */
class StripePaymentRequestController
{
    public function __construct(protected readonly AbstractStripeConfig $config)
    {
        //
    }

    /**
     * Set an address on basket
     *
     * @param SetAddress $request
     * @param BasketRepositoryInterface $basket
     * @return mixed
     */
    public function setShippingAddress(SetAddress $request, BasketRepositoryInterface $basket)
    {
        $basket->setAddress(\Mtc\Basket\Address::TYPE_SHIPPING, $request->only(array_keys(Address::$blueprint)));
        return response('ok');
    }

    /**
     * Create an order and payment
     * @param Request $request
     * @param PayableFactoryContract $payable_factory
     * @param InvoiceFactoryContract $invoice_factory
     * @return array
     * @throws \Exception
     */
    public function prepareOrder(
        Request $request,
        PayableFactoryContract $payable_factory,
        InvoiceFactoryContract $invoice_factory,
        InvoiceRepositoryContract $invoice_repository
    ): array {
        if (class_exists(BasketRepositoryInterface::class)) {
            $basket = App::make(BasketRepositoryInterface::class);
            if ($basket->getModel()->exists() == false) {
                throw new \Exception('Basket does not exist');
            }

            // Fill data from request
            App::make(StripePaymentRequestBasketFillInterface::class)->fill($request, $basket);

            // Build payable and invoice
            if (!$request->has('type')) {
                $request->merge(['type' => 'basket']);
            }
        }

        if ($request->filled('invoice_id')) {
            $invoice_repository = $invoice_repository->load($request->input('invoice_id'));
        }

        if (empty($invoice_repository->getId())) {
            $payable = $payable_factory->create($request);
            $invoice_repository = $invoice_factory->create($payable);
        }

        // Set up customer id and payment method
        \Stripe\Stripe::setApiKey($this->config->privateKey());
        $customer_id = App::make(Stripe::class)
            ->createCustomer($request->input('paymentMethod.id'), $request->input('payerEmail'));

        // Create payment object
        $intent = PaymentIntent::create([
            'description' => 'Payment on ' . config('app.name') . ' order ' . $invoice_repository->getReference(),
            'amount' => $invoice_repository->getOutstandingAmount() * 100,
            'currency' => $invoice_repository->getCurrency(),
            'customer' => $customer_id,
            'payment_method' => $request->input('paymentMethod.id'),
            'confirm' => true,
            'automatic_payment_methods' => [
                'enabled' => true,
                'allow_redirects' => 'never',
            ]
        ]);

        if ($intent->status === 'succeeded') {
            $this->attachPaymentMethodType($intent);
            $invoice_repository->savePayment([
                'provider' => 'stripe',
                'amount' => $intent->amount / 100,
                'reference' => $intent->id,
                'details' => $intent,
                'confirmed_at' => now(),
                'confirmation_status' => $this->config->confirmationStatus(),
            ]);
        }

        return [
            'success' => $intent->status === 'succeeded',
            'client_secret' => $intent->client_secret,
            'intent_id' => $intent->id,
            'invoice_id' => $invoice_repository->getId(),
        ];
    }

    /**
     * Confirm order has been paid
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice_repository
     * @return array
     */
    public function confirmOrder(
        Request $request,
        InvoiceRepositoryContract $invoice_repository
    ): array {
        $invoice_repository->load($request->input('invoice_id'));
        \Stripe\Stripe::setApiKey($this->config->privateKey());
        $intent = PaymentIntent::retrieve(request()->input('intent_id'));
        $this->attachPaymentMethodType($intent);

        if ($intent->status === 'succeeded') {
            $invoice_repository->savePayment([
                'provider' => 'stripe',
                'amount' => $intent->amount / 100,
                'reference' => $intent->id,
                'details' => $intent,
                'confirmed_at' => now(),
                'confirmation_status' => $this->config->confirmationStatus(),
            ]);
        }

        return [
            'success' => $intent->status === 'succeeded',
        ];
    }

    /**
     * Attach apple/google pay method details when available
     *
     * @param $intent
     */
    private function attachPaymentMethodType($intent)
    {
        if (empty($intent->charges->data[0]->payment_method_details->card->wallet->type)) {
            return;
        }

        $intent->payment_method_type = ucwords(str_replace(
            '_',
            ' ',
            $intent->charges->data[0]->payment_method_details->card->wallet->type
        ));
    }
}
