<?php

namespace Mtc\Stripe\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\URL;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Stripe\Stripe;
use Stripe\Error\Card;

/**
 * Class StripeController
 *
 * @package Mtc\Stripe\Http\Controllers
 */
class StripeController
{
    /**
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return array|\Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\Response
     */
    public function index(Request $request, InvoiceRepositoryContract $invoice)
    {
        \Stripe\Stripe::setApiKey(config('stripe.private_key'));
        $invoice->load($request->input('invoice_id'));

        if ($invoice->isPaid()) {
            return [
                'success' => true,
                'redirect' => URL::signedRoute('successful_payment', ['id' => $invoice->getId() ], null, config('app.env') === 'production'),
            ];
        }

        try {
            if ($request->has('payment_method_id')) {

                $customer_id = $this->getPaymentCustomerId($request, $invoice->getEmail());

                // Create payment intent
                $intent = \Stripe\PaymentIntent::create([
                    // Amount in pence / cents instead of eur/pounds
                    'description' => 'Payment on ' . config('app.name') . ' order ' . $invoice->getReference(),
                    'amount' => $invoice->getOutstandingAmountInCurrency() * 100,
                    'currency' => $invoice->getCurrency(),
                    'payment_method' => request()->input('payment_method_id'),
                    'customer' => $customer_id ?? null,
                    'confirmation_method' => 'manual',
                    'save_payment_method' => $this->shouldSavePaymentMethod($request),
                    'confirm' => $this->shouldChargeImmediately($request),
                ]);
            } elseif (request()->has('payment_intent_id')) {
                // Confirm payment intent
                $intent = \Stripe\PaymentIntent::retrieve(request()->input('payment_intent_id'));
                if ($this->shouldChargeImmediately($request)) {
                    $intent->confirm();
                }
            }
        } catch (Card $exception) {
            return response([
                'error' => $exception->getMessage()
            ], 422);
        }

        if (in_array($intent->status, ['requires_action', 'requires_source_action']) && $intent->next_action->type === 'use_stripe_sdk') {
            // Incomplete attempt, tell the client to handle the action
            return [
                'requires_action' => true,
                'payment_intent_client_secret' => $intent->client_secret
            ];
        }

        if ($intent->status === 'succeeded') {
            $invoice->savePayment([
                'provider' => 'stripe',
                'amount' => $intent->amount / 100,
                'reference' => $intent->id,
                'details' => $intent,
                'confirmed_at' => $this->shouldChargeImmediately($request) ? now() : null,
            ]);

            return [
                'success' => true,
                'redirect' => URL::signedRoute('successful_payment', ['id' => $invoice->getId() ], null, config('app.env') === 'production'),
            ];
        }

        return response([
            'error' => 'Invalid PaymentIntent status'
        ], 500);
    }

    /**
     * Check if card should be stored for the payment method
     *
     * @param $request
     * @return bool
     */
    protected function shouldSavePaymentMethod($request)
    {
        return $request->input('store_card') == 1;
    }

    /**
     * Get the customer ID for the order
     * @param $request
     * @return string|null
     */
    protected function getPaymentCustomerId($request, $email)
    {
        if ($request->input('payment_method_id') && $request->input('customer_id') !== 'false') {
            return $request->input('customer_id');
        }

        if ($this->shouldSavePaymentMethod($request) === false) {
            return null;
        }

        return $request->input('customer_id') !== 'false'
            ? $request->input('customer_id')
            : App::make(Stripe::class)->createCustomer($request->input('payment_method_id'), $email, $request->input('store_card'));
    }

    /**
     * Check if payment should be charged immediately
     *
     * @param $request
     * @return bool
     */
    protected function shouldChargeImmediately($request)
    {
        return config('checkout.deferred_payments') !== true || $request->input('strict') === 'true';
    }
}
