<?php

namespace Mtc\Stripe;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\HandlesRefunds;
use Mtc\Checkout\Contracts\HasDeferredCharges;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Checkout\PaymentForm;
use Mtc\Modules\Members\Classes\Auth;
use Stripe\PaymentIntent;

/**
 * Stripe Payment Gateway
 *
 * @package  Mtc\Stripe
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Stripe implements PaymentGateway, HasDeferredCharges, HandlesRefunds
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= config('stripe.minimal_transaction_amount')) {
            return false;
        }

        return App::make(config('stripe.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        \Stripe\Stripe::setApiKey(config('stripe.private_key'));

        return new PaymentForm('stripe-payment', 'vue-component', [
            'stripe_public_key' => config('stripe.public_key'),
            'invoice_id' => $invoice->getId(),
            'name' => __('stripe::stripe.payment_option_name'),
            'customer_cards' => $this->getCustomerCards(),
            'can_save_card' => $this->canSaveCards(),
            'customer' => $this->getCustomerToken(),
            'deferred_payments' => config('checkout.deferred_payments'),
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        // This is handled in StripeController
        return [];
    }

    /**
     * Check if payment can have a deferred charge
     *
     * @param Payment $payment
     * @return bool
     */
    public function allowDeferredCharge(Payment $payment)
    {
        if (config('checkout.deferred_payments') !== true) {
            return false;
        }

        \Stripe\Stripe::setApiKey(config('stripe.private_key'));
        $intent = PaymentIntent::retrieve($payment->reference);
        return $intent->status === 'requires_confirmation';
    }

    /**
     * Charge a payment that was set up as deferred
     *
     * @param Payment $payment
     * @return bool
     * @throws \Exception
     */
    public function chargeDeferredPayment(Payment $payment)
    {
        if (config('checkout.deferred_payments') !== true) {
            return false;
        }

        \Stripe\Stripe::setApiKey(config('stripe.private_key'));
        $intent = PaymentIntent::retrieve($payment->reference);
        $intent->confirm();
        if ($intent->status === 'succeeded') {
            return true;
        }

        if (in_array($intent->status, ['requires_action', 'requires_source_action'])) {
            $payment->triggerRescue();
            throw new \Exception('Card security error, requesting customer to authorise payment');
        }

        throw new \Exception($intent->status);
    }

    /**
     * Check if payment is refundable by payment gateway
     *
     * @param Payment $payment
     * @return bool|array
     */
    public function isRefundable(Payment $payment)
    {
        \Stripe\Stripe::setApiKey(config('stripe.private_key'));
        $intent = PaymentIntent::retrieve($payment->reference);
        if ($intent->charges->data[0]->refunds->total_count > 0) {
            $payment->refundable_amount = ($intent->charges->data[0]->amount - $intent->charges->data[0]->amount_refunded) / 100;
        }
        return $intent->charges->data[0]->refunded === false;
    }

    /**
     * Process a refund on payment
     *
     * @param Payment $payment
     * @param null $amount
     * @return bool|array
     */
    public function refund(Payment $payment, $amount = null)
    {
        \Stripe\Stripe::setApiKey(config('stripe.private_key'));
        $intent = PaymentIntent::retrieve($payment->reference);
        $refund = $intent->charges->data[0]
            ->refund([
                'amount' => $amount ? (int)($amount * 100) : null
            ]);

        if ($refund->status !== 'succeeded') {
            return false;
        }

        $amount = $refund->refunds->data[0]->amount ?? $refund->amount_refunded;
        return [
            'reference' => $refund->refunds->data[0]->id,
            'amount' => $refund->refunds->data[0]->amount / 100,
        ];
    }

    /**
     * Create customer record
     *
     * @param $payment_method_id
     * @param $email
     * @param int $store_card
     * @return string
     */
    public function createCustomer($payment_method_id, $email, $store_card)
    {
        $member = Auth::getLoggedInMember();
        $customer = \Stripe\Customer::create([
            'email' => $member->email ?? $email,
            'payment_method' => $payment_method_id,
        ]);

        if ($member->exists && $store_card) {
            $member->authenticators()
                ->create([
                    'provider' => 'stripe',
                    'key' => $customer->id,
                ]);
        }

        return $customer->id;

    }

    /**
     * Find customer saved cards
     * @return mixed
     */
    protected function canSaveCards()
    {
        return config('stripe.card_storage', false) && Auth::getLoggedInMember()->exists;
    }

    /**
     * Find customer saved cards
     * @return mixed
     */
    protected function getCustomerCards()
    {
        if (config('stripe.card_storage', false) === false) {
            return [];
        }

        $token = $this->getCustomerToken();
        if (!$token) {
            return [];
        }

        //retrieve stripe customer
        $cards = \Stripe\PaymentMethod::all([
            'customer' => $token,
            'type' => 'card'
        ]);

        return collect($cards->values()[1])
            ->map(function ($card) {
                return [
                    'id' => $card->id,
                    'brand' => ucfirst($card->card->brand),
                    'number' => '**** **** **** ' . $card->card->last4,
                    'expiry' => $card->card->exp_month . '/' . $card->card->exp_year
                ];
            });
    }

    /**
     * Get customers stripe token
     *
     * @return bool|string
     */
    protected function getCustomerToken()
    {
        if (config('stripe.card_storage', false) === false) {
            return false;
        }

        $member = Auth::getLoggedInMember();
        if (!$member->exists) {
            return false;
        }

        return $member->authenticators()->where('provider', 'stripe')->first()->key ?? false;
    }
}
