<?php

namespace Mtc\Stripe;

use Mtc\Modules\Members\Classes\Auth;
use Mtc\Modules\Members\Classes\Member;

/**
 * Class Customer
 *
 * Stripe functionality for Customer / Token handling
 *
 * @package Mtc\Stripe
 */
class Customer
{
    /**
     * Find the users stripe ID or create one
     *
     * @param Member $member
     * @param string $email
     * @return null|string
     */
    public function findOrCreateCustomerId($member, $email = '')
    {
        if ($this->isKnownCustomer($member)) {
            // Known customer means we have a customer ID already
            $customer_id = $this->getCustomersToken($member);
        } else {
            $customer = \Stripe\Customer::create([
                'email' => $member->email ?? $email
            ]);
            $customer_id = $customer->id;

            // Save customer_id to current user
            if ($member->exists) {
                $member->authenticators()->create([
                    'provider' => \Mtc\Stripe\Stripe::class,
                    'token' => $customer_id
                ]);
            }
        }
        return $customer_id;
    }

    /**
     * Retrieve customer cards from Stripe gateway
     * This method checks if user is logged in and has a stripe customer id
     * If user has customer ID we can check if there are cards bound with this user
     *
     * @return array|mixed
     */
    public function getCustomersCards()
    {
        /** @var Member|bool $user Auth facade will return user or false */
        $user = Auth::getLoggedInMember();

        // Check if user can actually have cards
        if (!self::isKnownCustomer($user)) {
            return null;
        }

        //Set Api Key with Stripe library
        \Stripe\Stripe::setApiKey(config('stripe.private_key'));

        try {
            $customer_id = self::getCustomersToken($user);
            $customer = \Stripe\Customer::retrieve($customer_id);
        } catch (\Exception $exception) {
            return null;
        }
        return $customer->sources->data;
    }

    /**
     * Get the customers stripe ID from saved token information against user
     *
     * @param Member $member
     * @return null
     */
    public function getCustomersToken($member)
    {
        return $member->authenticators()
            ->where('provider', self::class)
            ->first()
            ->key ?? null;
    }

    /**
     * Check if user is known and can use stripe token for pre-existing card
     *
     * @param Member $member
     * @return bool
     */
    public function isKnownCustomer($member)
    {
        // User must be logged in
        if (!$member) {
            return false;
        }

        // User must have a stripe token set on account
        return $member->authenticators()->where('provider', self::class)->exists();
    }

}
