<?php

namespace Mtc\Stripe;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;
use Mtc\Modules\Members\Classes\Auth;
use Mtc\Modules\Members\Classes\Member;
use Stripe\Charge;
use Stripe\Error\Card;
use Stripe\Error\InvalidRequest;

/**
 * Stripe Payment Gateway
 *
 * @package  Mtc\Stripe
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Stripe implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('stripe.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        \Stripe\Stripe::setApiKey(config('stripe.private_key'));
        return new PaymentForm('stripe-payment', 'vue-component', [
            'stripe_public_key' => config('stripe.public_key'),
            'invoice_id' => $invoice->getId(),
            'name' => __('stripe::stripe.payment_option_name'),
            'customer_cards' => $this->getCustomerCards(),
            'can_save_card' => $this->canSaveCards(),
            'customer' => $this->getCustomerToken(),
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        // This is handled in StripeController
        return [];
    }

    /**
     * Find customer saved cards
     * @return mixed
     */
    protected function canSaveCards()
    {
        return config('stripe.card_storage', false) && Auth::getLoggedInMember()->exists;
    }

    /**
     * Find customer saved cards
     * @return mixed
     */
    protected function getCustomerCards()
    {
        if (config('stripe.card_storage', false) === false) {
            return [];
        }

        $token = $this->getCustomerToken();
        if (!$token) {
            return [];
        }

        //retrieve stripe customer
        $cards = \Stripe\PaymentMethod::all([
            'customer' => $token,
            'type' => 'card'
        ]);

        return collect($cards->values()[1])
            ->map(function ($card) {
                return [
                    'id' => $card->id,
                    'brand' => ucfirst($card->card->brand),
                    'number' => '**** **** **** ' . $card->card->last4,
                    'expiry' => $card->card->exp_month . '/' . $card->card->exp_year
                ];
            });
    }

    /**
     * Get customers stripe token
     *
     * @return bool|string
     */
    protected function getCustomerToken()
    {
        if (config('stripe.card_storage', false) === false) {
            return false;
        }

        $member = Auth::getLoggedInMember();
        if (!$member->exists) {
            return false;
        }

        return $member->authenticators()->where('provider', 'stripe')->first()->key ?? false;
    }
}
