<?php
namespace Mtc\SiteNavigation;

use Illuminate\Database\Eloquent\Model;
use Mtc\Core\Facades\Media;

/**
 * Site Menu Image Eloquent model.
 *
 * @category Plugins
 * @package  Mtc_Site_Menu
 * @author   mtc. Ilya Berdnikov <ilya.berdnikov@mtcmedia.co.uk>
 * @date     2018-03-07
 */
class SiteNavigationImage extends Model
{
    /**
     * File mime types supported by image uploaders
     */
    const SUPPORTED_IMAGES_MIME_TYPES = [
        'image/png',
        'image/jpeg',
    ];

    /**
     * Classes associated table
     *
     * @var string
     */
    protected $table = 'site_navigation_images';

    /**
     * List of fillable variables
     *
     * @var array
     */
    protected $fillable = [
        'sitemenu_id',
        'url',
        'order',
    ];

    /**
     * Disables use of eloquent timestamps
     *
     * @var boolean
     */
    public $timestamps = true;

    /**
     * Extend model booting
     */
    protected static function boot()
    {
        parent::boot();

        static::deleting(function (self $image) {
            self::deleteUploadedMediaFiles($image->url, Media::getFolders('site_menu'));
        });
    }

    /**
     * Uploads image from $_FILES array by key defined in first parameter
     * Upload is made to folders specified in second parameter
     * Uses default core upload_image function
     *
     * @param string $file    Path to the file which needs to be uploaded
     * @param array  $folders Array of folders to upload the file to
     *
     * @return array
     */
    public static function uploadImage($file, $folders)
    {
        return upload_image($file, $folders);
    }

    /**
     * Used to delete images or videos uploaded using self::uploadImage()
     *
     * @param string $filename Filename which should be deleted
     * @param array  $folders  Array of folders which contain the filename
     *
     * @return void
     */
    public static function deleteUploadedMediaFiles($filename, $folders)
    {
        if (count($folders) > 0) {
            foreach ($folders as $folder) {
                $file_location = SITE_PATH . '/' . $folder['path'] . '/' . $filename;

                if (is_file($file_location)) {
                    unlink($file_location);
                }
            }
        }
    }

    /**
     * Get MIME type of a file
     *
     * @param string $file_path Path to the file to calculate MIME Type for
     *
     * @return string
     */
    public static function getMimeType($file_path)
    {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimetype = finfo_file($finfo, $file_path);
        finfo_close($finfo);

        return $mimetype;
    }
}
