<?php

namespace Mtc\SiteNavigation;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Cache;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Str;
use Kalnoy\Nestedset\NodeTrait;
use Mtc\SiteNavigation\Http\Presenters\MenuUrlPresenter;

/**
 * Class SiteNavigation
 *
 * @package Mtc\SiteNavigation
 */
class SiteNavigation extends Model
{
    use NodeTrait;

    /**
     * @var String Classes associated table
     */
    protected $table = 'site_navigation';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'url',
        'target',
        'sub_entry_type',
        'sub_entries',
        'css_class',
        'title',
        'description',
        'button_text',
        'button_link',
        'published',
    ];

    /**
     * Append attributes to json response
     * @var array
     */
    protected $appends = [
        'route_url',
    ];

    /**
     * Boot model
     */
    public static function boot()
    {
        parent::boot();

        static::creating(function (self $menu) {
            $menu->ensureUniqueSlug();
        });

        static::saved(function (self $menu){
            $menu->clearCache();
        });

        static::deleted(function (self $menu) {
            $menu->clearCache();
        });

        static::addGlobalScope(function ($query) {
            return $query->orderBy('_lft');
        });
    }

    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'slug';
    }


    /**
     * Relationship with images linked to current menu item
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function images()
    {
        return $this->hasMany(SiteNavigationImage::class, 'sitemenu_id')
            ->orderBy('order', 'ASC')
            ->orderBy('id', 'ASC');
    }

    /**
     * Pick up only published entries
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopePublished(Builder $query)
    {
        return $query->where('published', 1);
    }

    /**
     * URL builder
     *
     * @return MenuUrlPresenter
     */
    public function getRouteUrlAttribute()
    {
        return new MenuUrlPresenter($this);
    }

    /**
     * Make sure that slug for the navigation menu is unique
     */
    protected function ensureUniqueSlug()
    {
        $i = 0;
        $slug_base = $slug = Str::slug($this->name);
        while (self::query()->where('slug', $slug)->exists()) {
            $i++;
            $slug = "{$slug_base}-{$i}";
        }
        $this->slug = $slug;
    }

    /**
     * Clear menu cache
     *
     * @return void
     */
    public function clearCache()
    {
        $root = $this->getRoot();
        Cache::forget('navigation-menu-' . $root->slug);
        Cache::forget('navigation-menu-html-' . $root->slug);
    }

    /**
     * Get the root level menu of this leaf
     *
     * @return mixed
     */
    public function getRoot()
    {
        return self::query()
            ->whereIsRoot()
            ->where('_lft', '<=', $this->_lft)
            ->where('_rgt', '>=', $this->_rgt)
            ->first();
    }

    /**
     * Find model by slug with fallback of finding by id
     *
     * @param $slug
     * @return Model|null
     */
    public static function find($slug)
    {
        try {
            return self::query()
                ->where('slug', $slug)
                ->firstOrFail();
        } catch (ModelNotFoundException $exception) {
            if (is_numeric($slug)) {
                return self::query()->find($slug);
            }
        }
        return null;
    }

}

