<?php

namespace Mtc\SiteNavigation\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Mtc\SiteNavigation\Http\Resources\MenuList;
use Mtc\SiteNavigation\SiteNavigation;

/**
 * Class SiteNavigationController
 *
 * @package Mtc\SiteNavigation
 */
class NavigationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->page_meta['title'] = __('site_navigation::labels.manage_menus');
        return template('admin/site_navigation/index.twig', [
            'page_meta' => $this->page_meta,
            'menus' => SiteNavigation::query()->whereNull('parent_id')->latest()->paginate($request->input('per_page', 20))
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->page_meta['title'] = __('site_navigation::labels.create_menu');
        return template('admin/site_navigation/edit.twig', [
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $menu = SiteNavigation::query()
            ->create([
                'name' => $request->input('name'),
            ]);

        session()->flash('success', __('site_navigation::labels.created'));
        return redirect(route('navigation.show', [ $menu->slug ]));
    }

    /**
     * Display the specified resource.
     *
     * @param  SiteNavigation $menu
     * @return \Illuminate\Http\Response|MenuList
     */
    public function show(Request $request, SiteNavigation $menu)
    {
        $menu->load('children');

        if ($request->wantsJson()) {
            return new MenuList($menu);
        }
        $this->page_meta['title'] = __('site_navigation::labels.view_menu');
        return template('admin/site_navigation/show.twig', [
            'page_meta' => $this->page_meta,
            'menu' => $menu,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(SiteNavigation $navigation)
    {
        $this->page_meta['title'] = __('site_navigation::labels.edit_menu');
        return template('admin/site_navigation/edit.twig', [
            'page_meta' => $this->page_meta,
            'menu' => $navigation,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  SiteNavigation $menu
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update(Request $request, SiteNavigation $navigation)
    {
        $navigation->fill($request->input());
        $navigation->save();

        if ($request->wantsJson()) {
            $navigation->is_open = false;
            $navigation->edit = false;
            return $navigation;
        }
        session()->flash('success', __('site_navigation::labels.updated'));
        return redirect(route('navigation.show', [ $navigation->slug ]));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  Request $request
     * @param  SiteNavigation $menu
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, SiteNavigation $navigation)
    {
        $navigation->delete();
        if ($request->wantsJson()) {
            return response('success');
        }
        session()->flash('success', __('site_navigation::labels.removed'));
        return back();
    }

    /**
     * Add a child to the menu tree
     *
     * @param Request $request
     * @return MenuList
     */
    public function addChild(Request $request)
    {
        /** @var SiteNavigation $parent */
        $parent = SiteNavigation::query()->find($request->input('parent_id'));
        $child = $parent->children()
            ->create([
                'name' => $request->input('name'),
            ]);

        $parent->load('children');
        return new MenuList($parent);
    }

    /**
     * List menus based of filter
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function showList(Request $request)
    {
        $menu_list = SiteNavigation::query()->withDepth();

        if ($request->input('exclude', false)) {
            /** @var SiteNavigation $exclude */
            $exclude = SiteNavigation::query()->find($request->input('exclude', false));
            $root = $exclude->getRoot();

            $menu_list->whereDescendantOf($root)
                ->where('id', '!=', $exclude->id)
                ->whereNotDescendantOf($exclude);
        }

        /*
         * Check if we are restricting the list by depth
         * Location selection only allows root level menu so it needs filtering by depth of 0
         */
        if ($request->input('of_depth', false)) {
            $menu_list->having('depth', '=', $request->input('of_depth'));
        }

        $menu_list = $menu_list->orderBy('_lft')->get();

        /*
         * Reorder needs to have key/value map
         */
        if ($request->input('key_value_pair')) {
            $menu_list = $menu_list->keyBy('id')
                ->map(function ($menu) {
                    if (is_string($menu)) {
                        return $menu;
                    }

                    return str_repeat('-', $menu->depth) . ' ' . $menu->name;
                });
        }

        if ($request->input('prepend_root')) {
            $menu_list->put(0, trans('site_navigation::labels.root_level'));
        }

        // Return a prepared response
        // Doing this instead of collection as collection is already sorted and we don't want the order to change
        return response()->json($menu_list);
    }

    /**
     * Change the parent of a menu
     *
     * @param Request $request
     */
    public function changeParent(Request $request)
    {
        /** @var SiteNavigation $menu */
        $menu = SiteNavigation::find($request->input('menu_id'));

        if ($request->input('new_parent') == 0) {
            $menu->getRoot()->appendNode($menu);
        } else {
            /** @var SiteNavigation $parent */
            $parent = SiteNavigation::query()->find($request->input('new_parent'));
            $parent->appendNode($menu);
        }
    }

    /**
     * Reorder entries
     *
     * @param Request $request
     */
    public function reorder(Request $request)
    {
        foreach ($request->input('menus', []) as $key => $menu_data) {
            /** @var SiteNavigation $menu */
            $menu = SiteNavigation::query()->find($menu_data['id']);

            try {
                if (!empty($previous_menu)) {
                    $menu->afterNode($previous_menu)->save();
                }
            } catch (\Exception $exception) {
                // Ignore impossible move and proceed
                // e.g. the order of the first child has not changed causes the exception
            }

            // set this menu as previous so we can put next one besides this one
            $previous_menu = $menu;
        }

    }

    /**
     * Change the active state for menu entry
     *
     * @param SiteNavigation $menu
     * @return \Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\Response
     */
    public function toggleActive(SiteNavigation $menu)
    {
        $menu->published = !$menu->published;
        $menu->save();
        return response('success');
    }
}
