<?php

namespace Mtc\ShippingManager\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Support\Facades\Redirect;
use Mtc\ShippingManager\DeliveryCourier;
use Mtc\ShippingManager\Courier;
use Mtc\ShippingManager\CourierCollectLocation;

/**
 * Class DeliveryCourierController
 * @package Mtc\Plugins\ShippingManager\Classes\Http\Controllers\Admin
 */
class DeliveryCourierController extends Controller
{
    /**
     * @var
     */
    protected $check_mtc_role;

    /**
     * DeliveryCourierController constructor.
     *
     * @throws \Exception
     */
    public function __construct()
    {
        parent::__construct();

        $this->check_mtc_role = $this->admin_user->whereHas('roles', function ($query) {
            $query->where('name', 'mtc');
        })->count();

        self::checkInitialize();
    }

    /**
     * Check if plugin initialized (Have LIVE and test token)
     *
     * @throws \Exception
     */
    protected function checkInitialize()
    {
        if ($this->check_mtc_role) {
            $defined_settings = defined('DELIVERY_SERVICES_TOKEN_TEST') && defined('DELIVERY_SERVICES_TOKEN_LIVE');
            if ($defined_settings && empty(DELIVERY_SERVICES_TOKEN_TEST) && empty(DELIVERY_SERVICES_TOKEN_LIVE)) {
                Courier::initialize();
                $this::flashMessage('success', 'Delivery Client created successfully');
                Redirect::to('/admin/shipping-manager/delivery-couriers')->send();
            }
        }
    }

    /**
     * Display a listing of the resource.
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function index()
    {

        $self_action = url('/admin/shipping-manager/delivery-couriers');

        $couriers = Courier::getCourierServiceList();
        $available_couriers = Courier::filterAvailableCouriers($couriers, $this->check_mtc_role);

        $this->page_meta['page_name'] = 'Couriers List';

        return $this->twig->render('ShippingManager/admin/delivery_couriers/list.twig', [
            'page_meta' => $this->page_meta,
            'couriers' => $available_couriers,
            'self_action' => $self_action,
            'selected_tab_nav' => 'list',
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Request $request
     * @param $delivery_api_courier_id
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function edit(Request $request, $delivery_api_courier_id)
    {
        $update_action = dirname($request->getPathInfo());
        $store_action = url('/admin/shipping-manager/delivery-couriers');

        $courier_locations = CourierCollectLocation::all()
            ->map(function ($collect_location) {
                $address = $collect_location->address1;
                $address .= ($collect_location->address2 ? ', ' . $collect_location->address2 : '')
                    . ', ' . $collect_location->city . ', ' . $collect_location->postcode
                    . ', ' . $collect_location->country;
                $collect_location->address = $address;
                return $collect_location;
            })
            ->pluck('address', 'id');

        $delivery_courier = DeliveryCourier::query()
            ->where('api_courier_id', $delivery_api_courier_id)
            ->first();
        $api_delivery_courier = Courier::getSingleCourierServiceInformation($delivery_api_courier_id);


        if ($delivery_courier && (!$delivery_courier->id || !$delivery_courier->active) && !$this->check_mtc_role) {
            $api_delivery_courier = [];
        }

        $this->page_meta['page_name'] = 'Courier';

        return $this->twig->render('ShippingManager/admin/delivery_couriers/edit.twig', [
            'page_meta' => $this->page_meta,
            'courier' => $api_delivery_courier,
            'courier_information' => $delivery_courier,
            'courier_locations' => $courier_locations,
            'update_action' => $update_action,
            'store_action' => $store_action,
            'selected_tab_nav' => 'edit',
            'supported_label_formats' => Courier::$supported_label_formats
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param $delivery_api_courier_id
     * @return mixed
     */
    public function update(Request $request, $delivery_api_courier_id)
    {
        $validated = $this->validate($request, [
            'api_courier_id' => 'required',
            'collect_location_id' => 'required',
        ]);

        $delivery_courier = DeliveryCourier::query()->where('api_courier_id', $delivery_api_courier_id)->first();

        if ($delivery_courier) {
            $delivery_courier->fill($request->all())->save();
            session()->flash('success', 'Delivery Courier updated');
        }

        return redirect()->back();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @return mixed
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'api_courier_id' => 'required',
            'collect_location_id' => 'required',
        ]);

        DeliveryCourier::query()->create($request->all());

        session()->flash('success', 'Delivery Courier created');
        return redirect()->back();
    }
}
