# Revolut Payment Provider for Laravel

A Laravel package for integrating Revolut payment gateway into Mtc Pharmacy ( and MTC core ).

## Requirements

- PHP >= 8.2
- Laravel >= 9.0
- Composer
- Node.js & NPM (for frontend assets)

## Installation

1. Install the package via Composer:

```bash
composer require mtcmedia/revolut-payment-provider
```

2. Discover the package
```bash
php artisan package:discover --ansi
```

3. Publish the configuration file:

```bash
php artisan vendor:publish --provider="Mtc\RevolutPaymentProvider\RevolutServiceProvider" --tag="config"
```

4. Publish the database migrations:

```bash
php artisan vendor:publish --provider="Mtc\RevolutPaymentProvider\RevolutServiceProvider" --tag="database"
```

5. Run the migrations:

```bash
php artisan migrate
```

6. Publish the assets:

```bash
php artisan vendor:publish --provider="Mtc\RevolutPaymentProvider\RevolutServiceProvider" --tag="assets"
```

7. Seed the settings ( optional ):

```bash
php artisan db:seed --class="Database\Seeders\RevolutSettingsSeeder"
```

8. Add the following to your `.env` file:

```env
REVOLUT_ENABLED=0
REVOLUT_TEST_MODE=1


REVOLUT_SANDBOX_API_KEY=sk_....
REVOLUT_SANDBOX_PUBLIC_KEY=pk_....
REVOLUT_SANDBOX_API_URL=https://sandbox-merchant.revolut.com
REVOLUT_SANDBOX_WEBHOOK_SECRET=wsk_....
REVOLUT_SANDBOX_WEBHOOK_ID=[string from post]


REVOLUT_LIVE_API_KEY=sk_....
REVOLUT_LIVE_PUBLIC_KEY=pk_....
REVOLUT_LIVE_API_URL=https://merchant.revolut.com
REVOLUT_LIVE_WEBHOOK_SECRET=wsk_....
REVOLUT_LIVE_WEBHOOK_ID=[string from post]
```

9. Run the clearing of the caches

```bash
php artisan cache:clear; php artisan view:clear; php artisan route:clear; php artisan config:clear

```

10. Install and Compile the assets

```bash
npm install; npm run build
```


## Notes

1. Seeding the database and having the `.env `file populated the one that takes priority is the .env file always
In case there is a setting missing from the `.env` file, that will be automatically populated from the database
2. Due to the fact that the module is installed via composer there should not be a need anymore to update the `app/config.php` file
for the `'providers' => [...,\Mtc\RevolutPaymentProvider\RevolutServiceProvider::class,]'` due to the fact that it is defined in the composer.json as
```composer
"extra": {
    "laravel": {
      "providers": [
        "Mtc\\RevolutPaymentProvider\\RevolutServiceProvider"
      ],
      ...
    }
}

```
## Extra file definitions:
Make sure to have in `shop/admin/orders/manage.order.php` file the following line
```php
    if (!empty($order->payment_revolut)) {
        require_once($path . 'shop/admin/orders/templates/order.revolut.php');
    }
```

and the `shop/admin/orders/templates/order.revolut.php` defined as:
```php

<?php
use Mtc\RevolutPaymentProvider\Classes\RevolutPayment;

$transactions = RevolutPayment::query()->where("order_id", $order->getId())->orderByDesc("id")->get()->collect();
?>
<table>
<?php
    foreach ($transactions as $transaction) {
        ?>
            <tr>
                <td class="fieldname">Transaction Type</td>
                <td><strong><?php echo $transaction->type;?></strong></td>
            </tr>
            <tr>
                <td class="fieldname">Transaction ID</td>
                <td><?php echo $transaction->transaction_id;?></td>
            </tr>
            <tr>
                <td class="fieldname">Payment Amount</td>
                <td><?php echo $order->payment_revolut['amount'];?></td>
            </tr>

            <tr>
                <td class="fieldname">Payer Email</td>
                <td><?php echo $order->payment_revolut['member_email'];?></td>
            </tr>
            <tr>
                <td class="fieldName">Date</td>
                <td><?=$transaction->created_at ?></td>
            </tr>
            <tr>
                <td colspan="2"><hr/></td>
            </tr>
        <?php

}
?>
</table>


```

inside of the `shop/classes/class.order.php` 
```php
        // Revolut

        $sql = "SELECT * FROM `revolut_payments` WHERE `order_id`=:order_id";

        $db->run($sql, $params);

        if ($db->num_rows() > 0) {
            $this->payment_revolut = $db->fetch_array();
            $this->amount_paid = $this->payment_revolut['amount'];
            $this->order_ref = $this->payment_revolut['order_id'];
            $this->payment_gateway = "Revolut";
            $this->payment_icon = '/shop/admin/images/revolut.svg';
            return;
        }
```
Remember to upload the `revolut.svg` file in the mentioned path

Add the js file source for the revolut payment in `shop/templates/shop/basket/overview.twig`
```php
{% block head %}
    [..... ] 
    <script src="/public/js/revolut/revolut-pay.js"></script>
{% endblock %}    

```

## IMPORTANT
Remember to check the .htaccess file in case there is a removal of `www` to `non-www` so that the `.env` file can be updated to something similar to 
```env
APP_URL="https://${APP_DOMAIN}"
```
instead of
```env
APP_URL="https://www.${APP_DOMAIN}"
```

because of
```env
# Remove www throughout site.
RewriteCond %{HTTP_HOST} ^www\.(.*)$ [NC]
RewriteCond %{HTTP_HOST} !^.*www\.<subdomain>\.<domain>\.co\.uk [NC]
RewriteRule ^ https://%1%{REQUEST_URI} [L,R=301]
```

as that would cause the app to not post data to REVOLUT API endpoints

## Support

For support, please contact:
- Dan-Andrei Rotariu (danandrei.rotariu@mtc.co.uk)
- László Soltész (laszlo.soltesz@mtc.co.uk)
