<?php

namespace Mtcmedia\RevolutPaymentProvider;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Mtcmedia\RevolutPaymentProvider\Classes\RevolutPay;
use Mtcmedia\RevolutPaymentProvider\Services\RevolutService;
use Mtcmedia\RevolutPaymentProvider\Middleware\RevolutCorsMiddleware;

class RevolutServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register()
    {
        // Merge configuration
        $this->mergeConfigFrom(
            __DIR__.'/../config/revolut.php', 'revolut'
        );

        // Register the payment service as a singleton
        $this->app->singleton(RevolutService::class, function ($app) {
            return new RevolutService(
                config('revolut.api_key'),
                config('revolut.test_mode', true)
            );
        });

        // Register the CORS middleware
        $this->app['router']->aliasMiddleware('revolut.cors', RevolutCorsMiddleware::class);
    }

    /**
     * Bootstrap services.
     */
    public function boot()
    {
        // Publish configuration
        $this->publishes([
            __DIR__.'/../config/revolut.php' => config_path('revolut.php'),
        ], 'config');

        // Publish assets
        $this->publishes([
            __DIR__.'/../resources/js' => public_path('js/revolut'),
            __DIR__.'/../resources/css' => public_path('css/revolut'),
        ], 'assets');

        // Load routes
        $this->loadRoutesFrom(__DIR__.'/../routes/api.php');

        // Add the templates directory to the view paths
        $view_paths = config('view.paths', []);
        $view_paths[] = __DIR__.'/../resources/templates';
        config(['view.paths' => $view_paths]);

        // Register the template namespace
        $this->app['view']->addNamespace('revolut', __DIR__.'/../resources/templates');

        // Share configuration with views
        $this->app['view']->share('revolut_config', config('revolut'));

        // Listen for payment gateway loading
        $this->app['events']->listen('shop.checkout.load_payment_gateways', function ($basket) {
            $revolutPay = new RevolutPay(config('revolut'));
            return $revolutPay->form($basket);
        });
    }
}
