class RevolutPayment {
    constructor(config) {
        this.config = config;
        this.paymentForm = document.getElementById('revolut-payment-form');
        this.payButton = document.getElementById('revolut-pay-button');
        this.setupEventListeners();
        this.initializeRevolut();
    }

    initializeRevolut() {
        // Initialize Revolut.js
        this.revolut = new window.Revolut({
            publicKey: this.config.publicKey,
            locale: this.config.locale || 'en',
            style: {
                base: {
                    fontSize: '16px',
                    color: '#32325d',
                    fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
                }
            }
        });

        // Create card elements
        this.cardNumber = this.revolut.createCardElement('cardNumber');
        this.cardExpiry = this.revolut.createCardElement('cardExpiry');
        this.cardCvc = this.revolut.createCardElement('cardCvc');

        // Mount elements
        this.cardNumber.mount('#card-number-element');
        this.cardExpiry.mount('#card-expiry-element');
        this.cardCvc.mount('#card-cvc-element');
    }

    setupEventListeners() {
        // Handle payment method selection
        const paymentMethods = document.querySelectorAll('input[name="payment_method"]');
        paymentMethods.forEach(method => {
            method.addEventListener('change', (e) => this.handlePaymentMethodChange(e));
        });

        // Handle same as billing checkbox
        const sameAsBilling = document.getElementById('same-as-billing');
        if (sameAsBilling) {
            sameAsBilling.addEventListener('change', (e) => this.handleSameAsBilling(e));
        }

        // Handle form submission
        this.paymentForm.addEventListener('submit', (e) => this.handleSubmit(e));
    }

    handlePaymentMethodChange(event) {
        const method = event.target.value;
        const cardForm = document.getElementById('card-payment-form');

        if (method === 'card') {
            cardForm.style.display = 'block';
        } else {
            cardForm.style.display = 'none';
            // Handle Apple Pay or Google Pay
            this.handleDigitalWalletPayment(method);
        }
    }

    async handleSubmit(event) {
        event.preventDefault();
        this.payButton.disabled = true;
        this.payButton.textContent = 'Processing...';

        try {
            const paymentMethod = document.querySelector('input[name="payment_method"]:checked').value;

            if (paymentMethod === 'card') {
                await this.processCardPayment();
            } else {
                await this.processDigitalWalletPayment(paymentMethod);
            }
        } catch (error) {
            this.handleError(error);
        } finally {
            this.payButton.disabled = false;
            this.payButton.textContent = `Pay ${this.config.amount}`;
        }
    }

    async processCardPayment() {
        const { token, error } = await this.revolut.createToken();

        if (error) {
            throw new Error(error.message);
        }

        const response = await fetch('/revolut/payment', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            },
            body: JSON.stringify({
                token: token.id,
                amount: this.config.amount,
                currency: this.config.currency,
                billing: this.getBillingDetails(),
                shipping: this.getShippingDetails()
            })
        });

        const result = await response.json();

        if (!response.ok) {
            throw new Error(result.message || 'Payment failed');
        }

        this.handleSuccess(result);
    }

    getBillingDetails() {
        return {
            firstName: document.getElementById('billing-first-name').value,
            lastName: document.getElementById('billing-last-name').value,
            email: document.getElementById('billing-email').value,
            phone: document.getElementById('billing-phone').value,
            address: {
                line1: document.getElementById('billing-address').value,
                city: document.getElementById('billing-city').value,
                postalCode: document.getElementById('billing-postcode').value,
                country: document.getElementById('billing-country').value
            }
        };
    }

    getShippingDetails() {
        if (document.getElementById('same-as-billing').checked) {
            return this.getBillingDetails();
        }

        return {
            firstName: document.getElementById('shipping-first-name').value,
            lastName: document.getElementById('shipping-last-name').value,
            phone: document.getElementById('shipping-phone').value,
            address: {
                line1: document.getElementById('shipping-address').value,
                city: document.getElementById('shipping-city').value,
                postalCode: document.getElementById('shipping-postcode').value,
                country: document.getElementById('shipping-country').value
            }
        };
    }

    handleSameAsBilling(event) {
        const shippingFields = document.getElementById('shipping-address-fields');
        shippingFields.style.display = event.target.checked ? 'none' : 'block';
    }

    handleSuccess(result) {
        // Redirect to success page or show success message
        window.location.href = result.redirectUrl;
    }

    handleError(error) {
        // Show error message to user
        const errorElement = document.createElement('div');
        errorElement.className = 'alert alert-danger';
        errorElement.textContent = error.message;
        this.paymentForm.insertBefore(errorElement, this.payButton);
    }
}

// Initialize the payment form
document.addEventListener('DOMContentLoaded', () => {
    new RevolutPayment({
        publicKey: window.revolutConfig.publicKey,
        amount: window.revolutConfig.amount,
        currency: window.revolutConfig.currency,
        locale: window.revolutConfig.locale
    });
});
